# example_trigger.py -- An example Perforce trigger for enforcing workflow
#
# Richard Brooksby, Ravenbrook Limited, 2000-12-06.
#
# $Id: //info.ravenbrook.com/project/p4dti/version/0.5/code/replicator/example_trigger.py#2 $
#
#
# 1. INTRODUCTION
#
# This is an example Perforce trigger script for preventing changes
# to areas of the Perforce repository without certain defect tracker
# conditions being met.
#
# This document is intended for administrators of the Perforce Defect
# Tracking Integration (P4DTI) <http://www.ravenbrook.com/project/p4dti/>.
#
# This document is not confidential.
#
#
# 2. USING THE TRIGGER
#
# To use this trigger, first adapt the code to make it test the condition
# that you want to enforce.
#
# This example script insists that submissions fix to status "closed" at
# least one issue which has a priority of "critical".  This almost certainly
# isn't quite what you want, so you'll need to do some programming.
#
# You will also need to configure the P4DTI to replicate the fields that
# you want to check.  Most likely you'll want to check that the "priority"
# or "severity" or something similar is above a certain level, or that
# the "approval" field has been set by a manager, or something like that.
# You need to make sure the relevant fields are replicated.  See section
# 5.2, "Configuring the P4DTI software" of the Administrator's Guide
# [RB 2000-08-10, section 5.2].
#
# Be sure to read the "Triggers" section in chapter 6 of the "Perforce
# System Administrator's Guide" [Perforce 2000-10-11].  Then use the
# "p4 triggers" command to insert a line (without line breaks) like:
#
#         enforce-critical //depot/release/master/...
# "/usr/local/bin/python /whatever-path-to/your-trigger.py %serverport%
# root %client% %changelist%"
#
# Where "root" is a user which has access to Perforce without a password.
# (Or you could edit this script to include the password.)
#
# See "Perforce Defect Tracking Integration Administrator's Guide" for
# more information about this and other aspects of configuring the P4DTI
# [RB 2000-08-10].
#
# ---
#
# Copyright 2000 Ravenbrook Limited.  This document is provided "as is",
# without any express or implied warranty. In no event will the authors
# be held liable for any damages arising from the use of this document.
# You may make and distribute copies and derivative works of this
# document provided that (1) you do not charge a fee for this document or
# for its distribution, and (2) you retain as they appear all copyright
# and licence notices and document history entries, and (3) you append
# descriptions of your modifications to the document history.

import sys
import os
import marshal
import string

if len(sys.argv) != 5:
    sys.exit("python %s serverport user client changelist" % argv[0])

serverport, user, client, change = sys.argv[1:5]

os.environ["P4PORT"] = serverport
os.environ["P4USER"] = user
os.environ["P4CLIENT"] = client

def p4(command):
    stream = os.popen("p4 -G " + command, "r")
    results = []
    try:
        while 1:
            results.append(marshal.load(stream))
    except EOFError:
        pass
    if (len(results) == 1 and
        results[0].has_key('code') and
            results[0]["code"] == "error"):
        raise "Perforce Error", results[0]["data"]
    return results

fixes = p4("fixes -c %s" % change)

# Fixes look like this:
#  [{'Change': '1',
#    'Client': 'newton-lime',
#    'Date': '976102677',
#    'User': 'newton',
#    'code': 'stat',
#    'Job': '00003',
#    'Status': 'closed'},
#   ...]

# Enforce the rule that every submission must close at least one
# job with critical priority.  It might affect other jobs as well,
# but we don't care.

for fix in fixes:
    assert fix.has_key("Change") and fix["Change"] == change
    assert fix.has_key("Status")

    job = p4("job -o %s" % fix["Job"])[0]

    if (fix["Status"] == "closed" and
        job.has_key("Priority") and
        job["Priority"] == "critical"):
        sys.exit(0)

# No job was found that passed the test, so print a message to tell the
# submittor why we're refusing to accept the change.
print "Submissions to this codeline must fix at least one critical issue."
sys.exit(1)

# A. REFERENCES
#
# [Perforce 2000-10-11] "Perforce 2000.1 System Administrator's
# Guide"; Perforce Software; 2000-10-11; <http://www.perforce.com/
# perforce/doc.001/ manuals/p4sag/>, <ftp://ftp.perforce.com/
# /pub/perforce/ r00.1/doc/ manuals/p4sag/p4sag.pdf>.
#
# [RB 2000-08-10] "Perforce Defect Tracking Integration Administrator's
# Guide"; Richard Brooksby; Ravenbrook Limited; 2000-08-10
#
#
# B. DOCUMENT HISTORY
#
# 2000-12-06  RB  Created to solve Ravenbrook job job000043.
#
# 2000-12-15  RB  Fixed usage message to take script name from argv. 
# Detabbed to four spaces for consistency with the rest of our Python
# source.
