# sample_data.py -- put sample data for Perforce integration into a
# TeamTrack database.
# Gareth Rees, Ravenbrook Limited, 2000-09-04.
# $Id: //info.ravenbrook.com/project/p4dti/version/0.5/code/replicator/teamtrack_sample_data.py#2 $
#
# See "TeamTrack database schema extensions for integration with Perforce"
# <version/0.5/design/teamtrack-p4dti-schema/> for the database schema which this
# program generates sample data to match.
#
# Copyright 2000 Ravenbrook Limited.  This document is provided "as is",
# without any express or implied warranty. In no event will the authors
# be held liable for any damages arising from the use of this document.
# You may make and distribute copies and derivative works of this
# document provided that (1) you do not charge a fee for this document or
# for its distribution, and (2) you retain as they appear all copyright
# and licence notices and document history entries, and (3) you append
# descriptions of your modifications to the document history.

from perforce_sample_data import repository, random_filespecs
import random
import socket
import teamtrack
import time
import types

# Replicator id and Perforce server id for the generated relations.

localhost = socket.gethostname()
rid = 'case'
sid = 'p4_' + localhost

# Record types in the VCACTIONS table.  See "TeamTrack schema extensions
# for Perforce integration".

type_filespec = 1
type_fix = 2
type_change = 3

# The starting point for times generated by the program.

epoch = time.mktime((1998,1,1,0,0,0,0,0,0))

# Delete all records from the TS_VCACTIONS table.

def clear(server):
    records = server.query(teamtrack.table['VCACTIONS'],'')
    for r in records:
        server.delete_record(teamtrack.table['VCACTIONS'], r['ID'])

# Fill in the replicator id, Perforce server id, and jobname for each
# case in the TS_CASES table.

def case_fields(server, cases):
    for case in cases:
        case['P4DTI_RID'] = rid
        case['P4DTI_SID'] = sid
        case['P4DTI_JOBNAME'] = rid + '_' + str(case['ID'])
        case['P4DTI_REPLICATED'] = 0
        case.update()

# Add associations between cases and filespecs.

def assoc(server, cases):
    for case in cases:
        filespecs = random_filespecs()
        for f in filespecs:
            r = server.new_record(teamtrack.table['VCACTIONS'])
            r['TYPE'] = type_filespec
            r['CHAR1'] = rid
            r['CHAR2'] = sid
            r['TABLEID'] = teamtrack.table['CASES']
            r['RECID'] = case['ID']
            t = random.uniform(epoch, time.time())
            r['TIME1'] = t
            r['TIME2'] = t
            r['FILENAME'] = "{ 'filespec': '%s' }" % f
            r.add()

# The sample TeamTrack users (from the TeamShare demo) and the Perforce
# clients they use to edit files in the sample repository.

users = [
    { 'name': 'chad', 'id': 3, 'client': 'chad-elm' },
    { 'name': 'chris','id': 10, 'client': 'chris-larch' },
    { 'name': 'jill','id': 14, 'client': 'jill-pine' },
    { 'name': 'joe','id': 2, 'client': 'joe-beech' },
    { 'name': 'newton','id': 11, 'client': 'newton-holly' },
    { 'name': 'pat','id': 13, 'client': 'pat-lime' },
    { 'name': 'sarah','id': 12, 'client': 'sarah-willow' },
    ]

# Status values for Perforce jobs in the sample repository.

statuses = [ 'open', 'analyzed', 'changed', 'checked', 'released', 'closed' ]

def fixes_and_changes(server, cases):
    # Set all cases to "open"
    for case in cases:
        case['STATE'] = 0
    n_changes = len(cases) * 5          # Number of changes to make
    p4_change = 0                       # Perforce change number
    for i in range(n_changes):
        # Pick a time at which the change happens.
        t = epoch + (float(i) * (time.time() - epoch) / float(n_changes))
        # Pick a user who makes the change.
        u = random.choice(users)
        # Increase the Perforce change number.
        p4_change = p4_change + random.choice([1,1,1,1,1,2,2,3,4,5])
        # Make a change record.
        c = server.new_record(teamtrack.table['VCACTIONS'])
        c['TYPE'] = type_change
        c['CHAR1'] = rid
        c['CHAR2'] = sid
        c['INFO1'] = p4_change
        c['TIME1'] = t
        c['TIME2'] = t
        c['AUTHOR1'] = u['id']
        c['INFO2'] = 1                  # Submitted changelist.
        c['FILENAME'] = "{ 'decription': 'Automatically generated change comment for change %d.', 'client': '%s' }" \
                        % (p4_change, u['client'])
        c.add()
        # Pick k out of n cases for the change to affect.
        n = len(cases)
        k = random.choice([0,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,3])
        while n > 0 and k > 0:
            if (random.uniform(0,1) < float(k)/float(n)):
                case = cases[n-1]
                # Change the status keyword by moving it forward in the
                # statuses array with probability 90%
                if case['STATE'] == 0 or random.uniform(0,1) < 0.9:
                    case['STATE'] = case['STATE'] + 1
                else:
                    case['STATE'] = case['STATE'] - 1
                # Make a fix record.
                f = server.new_record(teamtrack.table['VCACTIONS'])
                f['TYPE'] = type_fix
                f['CHAR1'] = rid
                f['CHAR2'] = sid
                f['TABLEID'] = teamtrack.table['CASES']
                f['RECID'] = case['ID']
                f['INFO1'] = p4_change
                f['TIME1'] = t
                f['TIME2'] = t
                f['AUTHOR1'] = u['id']
                f['FILENAME'] = "{ 'status': '%s', 'client': '%s' }" \
                                % (statuses[case['STATE']], u['client'])
                f.add()
                k = k - 1
            n = n - 1

def run():
    print "Connecting to TeamTrack..."
    server = teamtrack.connect('joe','',localhost)
    print "Deleting all records from the VCACTIONS table..."
    clear(server)
    print "Getting cases..."
    cases = server.query(teamtrack.table['CASES'], '')
    print "Filling in P4DTI fields for all cases..." 
    case_fields(server, cases)
    print "Associating filespecs with each case..."
    assoc(server, cases)
    print "Adding fixes and changes..."
    fixes_and_changes(server, cases)

# B. Document History
# 
# 2000-09-06 GDR Created.
# 
# 2000-09-07 GDR Add files directly to TeamTrack rather than writing to a file
# in CSV format.
# 
# 2000-09-11 GDR Use teamtrack.table instead of teamtrack.tables.  Use
# replicator and server ids consistent with the replicator defaults.
# 
# 2000-09-12 GDR Put a status keyword into the fixes relation, not an integer
# (this fixes the bug reported by Larry Fish in his e-mail 2000-09-12 02:11:29
# GMT).
# 
# 2000-09-25 GDR Repository description and random_filespecs function moved to
# perforce_sample_data, which is a more logical place for it.
