// TSItem.cpp: implementation of the TSItem class.
//
/////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include "TSItem.h"
#include "TSServer.h"
#include <algorithm>
#include "TSUtility.h"

// This class defines the base interface for both primary and auxiliary
//  items. A base class makes sense since these items have several shared
//  characteristics. This class should not be instantiated directly.
//  Therefore, the constructor is protected. Please see the TSAuxiliaryItem
//  and TSPrimaryItem derived classes. 

TSItem::TSItem( TSServer& server, int nTableId /*=0*/, int nItemId /*=0*/ )
  : m_attachmentList( ),
    m_changeList    ( ),
    m_fieldList     ( ),
    m_linkList      ( ),
    m_nMode         ( 0 ),
    m_nSectionMask  ( TS_SECTMASK_ALL_EXCEPT_FILE_CONTENTS ),
    m_nTransitionId ( 0 ),
    TSRecordRef     ( server, nTableId, nItemId )
{
}

TSItem::TSItem( const TSItem& that )
  : m_nMode         ( that.m_nMode ),
    m_nSectionMask  ( that.m_nSectionMask ),
    m_nTransitionId ( that.m_nTransitionId ),
    TSRecordRef     ( that )
{
  TSAttachmentList::const_iterator itAttach = that.m_attachmentList.begin();
  for ( ; itAttach != that.m_attachmentList.end(); itAttach++ )
  {
    TSAttachment* pAttachment = new TSAttachment( **itAttach );
    m_attachmentList.push_back( pAttachment );
  }

  TSChangeList::const_iterator itChange = that.m_changeList.begin();
  for ( ; itChange != that.m_changeList.end(); itChange++ )
  {
    TSChangeHistory* pChangeHistory = new TSChangeHistory( **itChange );
    m_changeList.push_back( pChangeHistory );
  }
  
  TSDisplayFieldList::const_iterator itField = that.m_fieldList.begin();
  for ( ; itField != that.m_fieldList.end(); itField++ )
  {
    TSDisplayField* pDisplayField = new TSDisplayField( **itField );
    m_fieldList.push_back( pDisplayField );
  }

  TSItemLinkList::const_iterator itLink = that.m_linkList.begin();
  for ( ; itLink != that.m_linkList.end(); itLink++ )
  {
    TSItemLink* pItemLink = new TSItemLink( **itLink );
    m_linkList.push_back( pItemLink );
  }
}

TSItem::~TSItem()
{
  TSAttachmentList::iterator itAttach = m_attachmentList.begin();
  for ( ; itAttach != m_attachmentList.end(); itAttach++ )
  {
    delete *itAttach;
  }
  TSChangeList::iterator itChange = m_changeList.begin();
  for ( ; itChange != m_changeList.end(); itChange++ )
  {
    delete *itChange;
  }
  TSDisplayFieldList::iterator itField = m_fieldList.begin();
  for ( ; itField != m_fieldList.end(); itField++ )
  {
    delete *itField;
  }
  TSItemLinkList::iterator itLink = m_linkList.begin();
  for ( ; itLink != m_linkList.end(); itLink++ )
  {
    delete *itLink;
  }
}

TSItem& TSItem::operator = ( const TSItem& that )
{
  TSRecordRef::operator = ( that );

  TSDeleteStdList< TSAttachment    >( m_attachmentList );
  TSDeleteStdList< TSChangeHistory >( m_changeList );
  TSDeleteStdList< TSDisplayField  >( m_fieldList );
  TSDeleteStdList< TSItemLink      >( m_linkList );

  TSAttachmentList::const_iterator itAttach = that.m_attachmentList.begin();
  for ( ; itAttach != that.m_attachmentList.end(); itAttach++ )
  {
    TSAttachment* pAttachment = new TSAttachment( **itAttach );
    m_attachmentList.push_back( pAttachment );
  }

  TSChangeList::const_iterator itChange = that.m_changeList.begin();
  for ( ; itChange != that.m_changeList.end(); itChange++ )
  {
    TSChangeHistory* pChangeHistory = new TSChangeHistory( **itChange );
    m_changeList.push_back( pChangeHistory );
  }
  
  TSDisplayFieldList::const_iterator itField = that.m_fieldList.begin();
  for ( ; itField != that.m_fieldList.end(); itField++ )
  {
    TSDisplayField* pDisplayField = new TSDisplayField( **itField );
    m_fieldList.push_back( pDisplayField );
  }

  TSItemLinkList::const_iterator itLink = that.m_linkList.begin();
  for ( ; itLink != that.m_linkList.end(); itLink++ )
  {
    TSItemLink* pItemLink = new TSItemLink( **itLink );
    m_linkList.push_back( pItemLink );
  }

  m_nMode          = that.m_nMode;
  m_nSectionMask   = that.m_nSectionMask;
  m_nTransitionId  = that.m_nTransitionId;
  return *this;
}

int TSItem::AddAttachment( int nType, const TSString& sLabel, const TSString& sText )
{
  if ( !IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  return m_server.TSItemAddAttachment( *this, nType, sLabel, sText );
}

int TSItem::AddItemLinkById( int nDestTableId, int nDestItemId, int nLinkType )
{
  if ( !IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  return m_server.TSItemAddItemLink( *this, nDestTableId, nDestItemId, nLinkType, 0, 0 );
}

int TSItem::AddItemLinkByNumber( int nDestProjectId, int nDestItemNumber, int nLinkType )
{
  if ( !IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  return m_server.TSItemAddItemLink( *this, 0, 0, nLinkType, nDestProjectId, nDestItemNumber );
}

namespace {

  class FindRecRefByID
  {
  public:
    FindRecRefByID( int id ) : m_id( id ) {}
    bool operator()( const TSRecordRef* pRecRef )
    {
      return pRecRef->GetItemId() == m_id;
    }
    int m_id;
  };

}

int TSItem::DeleteAttachment( TSAttachmentList::iterator it )
{
  if ( m_attachmentList.size() > 0 )
  {
    const int id = (*it)->GetItemId();
    FindRecRefByID finder( id );
    TSAttachmentList::iterator foundIt = std::find_if( m_attachmentList.begin(), m_attachmentList.end(), finder );
    if ( foundIt != m_attachmentList.end() )
    {
      if ( m_server.TSItemDeleteAttachment( id, false ))
      {
        delete *foundIt;
        m_attachmentList.erase( foundIt );
        return TS_OK;
      }
    }
    else
    {
      TSSetLastError( TS_ITEM_NOT_FOUND );
    }
  }
  else
  {
    TSSetLastError( TS_ITEM_NOT_FOUND );
  }
  return TSGetLastError();
}

int TSItem::DeleteItemLink( TSItemLinkList::iterator it, bool bDeleteBothSides /*=true*/ )
{
  if ( m_linkList.size() > 0 )
  {
    const int id = (*it)->GetItemId();
    FindRecRefByID finder( id );
    TSItemLinkList::iterator foundIt = std::find_if( m_linkList.begin(), m_linkList.end(), finder );
    if ( foundIt != m_linkList.end() )
    {
      if ( m_server.TSItemDeleteAttachment( id, bDeleteBothSides ))
      {
        delete *foundIt;
        m_linkList.erase( foundIt );
        return TS_OK;
      }
    }
    else
    {
      TSSetLastError( TS_ITEM_NOT_FOUND );
    }
  }
  else
  {
    TSSetLastError( TS_ITEM_NOT_FOUND );
  }
  return TSGetLastError();
}

int TSItem::FinishSubmit()
{
  if ( m_nMode == TRANSITION_STARTED )
  {
    int nItemNumber = 0;
    int nResult = m_server.TSItemFinishSubmit( *this, nItemNumber );
    if ( nResult == TS_OK )
    {
      m_nMode = NOT_IN_TRANSITION;
    }
    return nResult;
  }
  else
  {
    return TS_INCORRECT_MODE;
  }
}

bool TSItem::IsValid() const
{
  if ( m_nItemId < 1 || m_nTableId < 1 )
  {
    return false;
  }
  return true;
}

int TSItem::Read()
{
  TSSetLastError( TS_ERROR );
  return TSGetLastError();
}

int TSItem::SetItemId( int nNewItemId )
{
  if ( m_nMode == TRANSITION_STARTED )
  { 
    return TS_NOT_UPDATEABLE;
  }
  else
  {
    m_nItemId = nNewItemId;
    return TS_OK;
  }
}

int TSItem::SetTableId( int nNewTableId )
{
  if ( m_nMode == TRANSITION_STARTED )
  { 
    return TS_NOT_UPDATEABLE;
  }
  else
  {
    m_nTableId = nNewTableId;
    return TS_OK;
  }
}

int TSItem::StartSubmit()
{
  if ( m_nMode == NOT_IN_TRANSITION )
  {
    int nResult = m_server.TSItemStartSubmit( *this );
    if ( nResult == TS_OK )
    {
      m_nMode = TRANSITION_STARTED;
    }
    return nResult;
  }
  else
  {
    return TS_INCORRECT_MODE;
  }
}
