// TSItem.h: interface for the TSItem class.
//
/////////////////////////////////////////////////////////////////////////

#ifndef TSITEM_H_
#define TSITEM_H_

#include "TSAttachment.h"
#include "TSChangeHistory.h"
#include "TSDisplayField.h"
#include "TSItemLink.h"
#include "TSRecordRef.h"
#include "TSString.h"

#pragma warning( push, 1 )
#include "DisableSTLWarnings.h"
#include <list>
#pragma warning( pop )

#pragma warning( disable: 4710 )  // inlined function not inlined

const int TRANSITION_STARTED = 1;
const int NOT_IN_TRANSITION  = 0;

// This class defines the base interface for both primary and auxiliary
//  items. A base class makes sense since these items have several shared
//  characteristics. This class should not be instantiated directly.
//  Therefor, the constructor is protected. Please see the TSAuxiliaryItem
//  and TSPrimaryItem derived classes. 

class AFX_EXT_CLASS TSItem : public TSRecordRef
{
  friend class TSServer;
  public: 
    virtual ~TSItem();

    // Accessor methods:
    const TSAttachmentList&    GetAttachmentList() const { return m_attachmentList; }
          TSAttachmentList&    GetAttachmentList()       { return m_attachmentList; }
    const TSChangeList&        GetChangeList() const     { return m_changeList; }
          TSDisplayFieldList&  GetFieldList()            { return m_fieldList; }
    const TSDisplayFieldList&  GetFieldList() const      { return m_fieldList; }
          TSItemLinkList&      GetLinkList()             { return m_linkList; }
    const TSItemLinkList&      GetLinkList() const       { return m_linkList; }
    /* -------------------------------------------------------------------
    The bit masks describing the sections that will be obtained when the
     following accessor methods are called are:
    	TS_SECTMASK_STANDARD.......Standard Fields
    	TS_SECTMASK_USER...........User Fields
    	TS_SECTMASK_ADVANCED.......Advanced Fields
    	TS_SECTMASK_MANAGER........Manager Fields
    	TS_SECTMASK_SYSTEM.........System Fields
    	TS_SECTMASK_NOTES..........Notes
    	TS_SECTMASK_FILES..........File Attachments
    	TS_SECTMASK_URLS...........URL Attachments
    	TS_SECTMASK_LINKS..........Item Links 
    	TS_SECTMASK_CHANGES........Change History
    	TS_SECTMASK_NOTE_CONTENTS..Include contents of notes with notes
    	TS_SECTMASK_FILE_CONTENTS..Include contents of files with files
    ----------------------------------------------------------------------- */
    void  ClearSectionMaskBits( int nBitsToClear ) { m_nSectionMask &= ~nBitsToClear; }
    int   GetSectionMask() const                   { return m_nSectionMask; }
    void  SetSectionMask( int nNewSectionMask )    { m_nSectionMask = nNewSectionMask; }
    void  SetSectionMaskBits( int nBitsToSet )     { m_nSectionMask |= nBitsToSet; }
    
    // These Set accessors allow modification when m_nMode is not == to TS_TRANSITION_STARTED.
    virtual int SetItemId( int nNewItemId );
    virtual int SetTableId( int nNewTableId );

    // The AddAttachment method adds an attachment
    //    ( TS_ATTACHATTRIB_NOTE - Note,
    //      TS_ATTACHATTRIB_URL  - URLs, or
    //      TS_ATTACHATTRIB_FILE -  Files) to this item.
    // The sText argument should be --
    //      for files: the full path to the file that will be read,
    //      for notes: the note content, or
    //      for URLs:  the complete URL.
    // This requires a roundtrip to the server.
    int AddAttachment( int nType, const TSString& sLabel, const TSString& sText );

    // The AddItemLinkById method adds a link to this item. Either the table and
    //  item id, or the project id and item number should be set before using. The
    //  nLinkType argument is a bit flag argument with bit values defined as:
    //      TS_STORE_ACTION_TWOWAYLINK    = 0x0100,
    //      TS_STORE_ACTION_SOURCETRIGGER = 0x0200, and
    //      TS_STORE_ACTION_DESTTRIGGER   = 0x0400.
    // This requires a roundtrip to the server.
    int AddItemLinkById( int nDestTableId, int nDestItemId, int nLinkType );

    // The AddItemLinkByNumber method adds a link to this item. Either the table
    //  and item id, or the project id and item number should be set before using.
    //  The nLinkType argument is a bit flag argument with bit values defined as:
    //      TS_STORE_ACTION_TWOWAYLINK    = 0x0100,
    //      TS_STORE_ACTION_SOURCETRIGGER = 0x0200, and
    //      TS_STORE_ACTION_DESTTRIGGER   = 0x0400.
    // This requires a roundtrip to the server.
    int AddItemLinkByNumber( int nDestProjectId, int nDestItemNumber, int nLinkType ) ;

    // The DeleteAttachment method deletes the attachment from this item and from
    //  the attachment list.
    // This requires a roundtrip to the server.
    int DeleteAttachment( TSAttachmentList::iterator it );

    // The DeleteItemLink method deletes the item link from this item and from
    //  the link list. The bDeleteBothSides argument specifies whether to also
    //  delete a link back to this item if it exists.
    // This requires a roundtrip to the server.
    int DeleteItemLink( TSItemLinkList::iterator it, bool bDeleteBothSides = true );

    // The FinishSubmit method is used to add the item to the database after
    //  field values in the field list (obtained from StartSubmit) have been set.
    //  This method will only send fields that have been modified back to the
    //  server.
    // This requires a roundtrip to the server.
    virtual int FinishSubmit();

    // The StartSubmit method should be used to fill in this item as appropriate
    //  for a submit. StartSubmit populates the item with defaults and orders
    //  fields as they would be for a submit, including filling out field selection
    //  lists. The three other member lists of a TSItem: changeList, attachmentList
    //  and linkList, will be emptied. The table id must be preset.
    // This requires a roundtrip to the server.
    virtual int StartSubmit();

  protected:
    // Constructors/destructors/operator =
    // The constructors will set all bit masks, except TS_SECTMASK_FILE_CONTENTS.
    TSItem( TSServer& server, int nTableId = 0, int nItemId = 0 );
    TSItem( const TSItem& that );
    TSItem& operator = ( const TSItem& that );

    // The IsValid method is called from the server and in Read to make
    //  sure that table id and item id have been set.
    virtual bool IsValid() const;

    TSAttachmentList    m_attachmentList; // List of TSAttachment objects
    TSChangeList        m_changeList;     // List of TSChangeHistory objects,
                                                  //   ordered by user preference.
    TSDisplayFieldList  m_fieldList;      // List of TSDisplayField objects
    TSItemLinkList      m_linkList;       // List of TSItemLink objects
    int	m_nMode;        // Internal variable to keep track of status
    int m_nSectionMask; // Mask used on Read to determine how & which lists to fill
    int m_nTransitionId;// Used to keep track of transition

  private:
    virtual int Read();

};

typedef std::list< TSItem* > TSItemList;

#endif
