// TSPrimaryItem.cpp: implementation of the TSPrimaryItem class.
//
/////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include "TSPrimaryItem.h"
#include "TSServer.h"

// This class defines a primary item.

TSPrimaryItem::TSPrimaryItem( TSServer& server, int nTableId /*=0*/, int nItemId /*=0*/ )
  : m_nProjectId   ( 0 ),
    m_nItemNumber  ( 0 ),
    m_nStateId     ( 0 ),
    m_sItemType    ( "" ),
    TSItem         ( server, nTableId, nItemId )
{
}

TSPrimaryItem::TSPrimaryItem( TSServer& server, const TSProject& project, int nItemNumber /*=0*/ )
  : m_nProjectId   ( project.GetProjectId()),
    m_nItemNumber  ( nItemNumber ),
    m_nStateId     ( 0 ),
    m_sItemType    ( "" ),
    TSItem         ( server, project.GetPrimaryTableId(), 0 )
{
}

TSPrimaryItem::TSPrimaryItem( const TSPrimaryItem& that )
  : m_nProjectId   ( that.m_nProjectId ),
    m_nItemNumber  ( that.m_nItemNumber ),
    m_nStateId     ( that.m_nStateId ),
    m_sItemType    ( that.m_sItemType ),
    TSItem         ( that )
{
}

TSPrimaryItem::~TSPrimaryItem()
{
}

TSPrimaryItem& TSPrimaryItem::operator = ( const TSPrimaryItem& that )
{
  TSItem::operator = ( that );
  m_nItemNumber    = that.m_nItemNumber;
  m_nProjectId     = that.m_nProjectId;
  m_nStateId       = that.m_nStateId;
  m_sItemType      = that.m_sItemType;
  return *this;
}

int TSPrimaryItem::CancelTransition()
{
  if ( m_nMode == TRANSITION_STARTED )
  {
    if ( !IsValid() )
    {
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    int nResult = m_server.TSPrimaryItemCancelTransition( *this );
    if ( nResult == TS_OK ) // TO DO: Which return values release the lock?
    {
      m_nMode = NOT_IN_TRANSITION;
      m_nItemId = 0;
    }
    TSSetLastError( nResult );
  }
  else
  {
    TSSetLastError ( TS_INCORRECT_MODE );
  }
  return TSGetLastError();
}

int TSPrimaryItem::FinishSubmit()
{
  if ( m_nMode == TRANSITION_STARTED )
  {
    int nItemNumber = 0;
    int nResult = m_server.TSItemFinishSubmit( *this, nItemNumber );
    if ( nResult == TS_OK )
    {
      m_nMode = NOT_IN_TRANSITION;
      m_nItemNumber = nItemNumber;
    }
    return nResult;
  }
  else
  {
    return TS_INCORRECT_MODE;
  }
}

int TSPrimaryItem::FinishTransition()
{
  if ( m_nMode == TRANSITION_STARTED )
  {
    if ( !IsValid() )
    {
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    int nResult = m_server.TSPrimaryItemFinishTransition( *this );
    if ( nResult == TS_OK ) // TO DO: Which return values release the lock?
    {
      m_nMode = NOT_IN_TRANSITION;
    }
    TSSetLastError( nResult );
  }
  else
  {
    TSSetLastError ( TS_INCORRECT_MODE );
  }
  return TSGetLastError();
}

int TSPrimaryItem::GetTransitionList( TSTransitionList& results )
{
  if ( !IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  return m_server.TSPrimaryItemGetTransitionList( *this, results );
}

bool TSPrimaryItem::IsValid() const
{
  if (( m_nItemId > 0 && m_nTableId > 0 ) || ( ( m_nProjectId > 0 || m_nTableId > 0 ) && m_nItemNumber > 0 ))
  { // You can use table id and item number, but it is not guaranteed to be unique
    return true;
  }

  return false;
}

int TSPrimaryItem::Read()
{
  if ( !IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  if ( m_nMode == TRANSITION_STARTED )
  {
    TSSetLastError( TS_INCORRECT_MODE );
    return TSGetLastError();
  }
  return m_server.TSPrimaryItemRead( *this );
}

int TSPrimaryItem::ReadByNumber( int nProjectId, int nItemNumber )
{
  if ( m_nMode == TRANSITION_STARTED )
  {
    TSSetLastError( TS_INCORRECT_MODE );
    return TSGetLastError();
  }
  this->SetProjectId( nProjectId );
  this->SetItemNumber( nItemNumber );
  return Read();
}

int TSPrimaryItem::Receive( TSSocket* socket )
{
  if ( socket->ReceiveInt( &m_nItemNumber ) != TS_OK )
  {
    return TSGetLastError();
  }
  if ( socket->ReceiveInt( &m_nProjectId ) != TS_OK )
  {
    return TSGetLastError();
  }
  if ( socket->ReceiveInt( &m_nStateId ) != TS_OK )
  {
    return TSGetLastError();
  }
  if ( socket->ReceiveString( &m_sItemType ) != TS_OK )
  {
    return TSGetLastError();
  }
  return TS_OK;
}

int TSPrimaryItem::SetItemId( int nNewItemId )
{
  if ( m_nMode == TRANSITION_STARTED )
  { 
    TSSetLastError( TS_NOT_UPDATEABLE );
    return TSGetLastError();
  }
  else
  {
    m_nItemId = nNewItemId;
    m_nItemNumber = 0;
    return TS_OK;
  }
}

int TSPrimaryItem::SetItemNumber( int nNewItemNumber )
{
  if ( m_nMode == TRANSITION_STARTED )
  { 
    TSSetLastError( TS_NOT_UPDATEABLE );
    return TSGetLastError();
  }
  else
  {
    m_nItemNumber = nNewItemNumber;
    m_nItemId = 0;
    return TS_OK;
  }
}

int TSPrimaryItem::SetProjectId( int nNewProjectId )
{
  if ( m_nMode == TRANSITION_STARTED )
  {
    TSSetLastError( TS_NOT_UPDATEABLE );
    return TSGetLastError();
  }
  else
  {
    m_nProjectId = nNewProjectId;
    return TS_OK;
  }
}

int TSPrimaryItem::SetTableId( int nNewTableId )
{
  if ( m_nMode == TRANSITION_STARTED )
  {
    TSSetLastError( TS_NOT_UPDATEABLE );
    return TSGetLastError();
  }
  else
  {
    m_nTableId = nNewTableId;
    return TS_OK;
  }
}

int TSPrimaryItem::StartSubmit()
{
  if ( m_nMode == NOT_IN_TRANSITION )
  {
    if ( m_nProjectId == 0 )
    { // Must have a project id for start submit
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    int nResult = m_server.TSItemStartSubmit( *this );
    if ( nResult == TS_OK )
    {
      m_nMode = TRANSITION_STARTED;
    }
    TSSetLastError( nResult );
  }
  else
  {
    TSSetLastError( TS_INCORRECT_MODE );
  }
  return TSGetLastError();
}

int TSPrimaryItem::StartSubmitIntoProject( int nProjectId )
{
  if ( m_nMode == NOT_IN_TRANSITION )
  {
    SetProjectId( nProjectId );
    SetTableId( 0 );
    SetItemId( 0 );
    return StartSubmit();
  }
  else
  {
    TSSetLastError( TS_INCORRECT_MODE );
    return TSGetLastError();
  }
}

int TSPrimaryItem::StartSubmitIntoProject( TSProject& project )
{
  if ( m_nMode == NOT_IN_TRANSITION )
  {
    SetProjectId( project.GetItemId() );
    SetTableId( 0 );
    SetItemId( 0 );
    return StartSubmit();
  }
  else
  {
    TSSetLastError( TS_INCORRECT_MODE );
    return TSGetLastError();
  }
}

int TSPrimaryItem::StartTransition( int nTransitionId )
{
  if ( m_nMode == NOT_IN_TRANSITION )
  {
    if ( !IsValid() )
    {
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    int nResult = m_server.TSPrimaryItemStartTransition( *this, nTransitionId );
    if ( nResult == TS_OK )
    {
      m_nMode = TRANSITION_STARTED;
    }
    TSSetLastError( nResult );
  }
  else
  {
    TSSetLastError( TS_INCORRECT_MODE );
  }
  return TSGetLastError();
}

int TSPrimaryItem::StartTransition( TSString sTransitionName )
{
  if ( m_nMode == NOT_IN_TRANSITION )
  {
    if ( !IsValid() )
    {
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    int nResult = m_server.TSPrimaryItemStartTransition( *this, 0, sTransitionName );
    if ( nResult == TS_OK )
    {
      m_nMode = TRANSITION_STARTED;
    }
    TSSetLastError( nResult );
  }
  else
  {
    TSSetLastError( TS_INCORRECT_MODE );
  }
  return TSGetLastError();
}

int TSPrimaryItem::StartTransition( const TSTransition& transitionItem )
{
  if ( m_nMode == NOT_IN_TRANSITION )
  {
    if ( !IsValid() )
    {
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    int nResult = m_server.TSPrimaryItemStartTransition( *this, transitionItem.GetItemId() );
    if ( nResult == TS_OK )
    {
      m_nMode = TRANSITION_STARTED;
    }
    TSSetLastError( nResult );
  }
  else
  {
    TSSetLastError( TS_INCORRECT_MODE );
  }
  return TSGetLastError();
}

TSString TSPrimaryItem::StringDump( TSString sIndentation )
{
  char tmpBuf[64];
  TSString sSpacer = sIndentation + "  ";
  TSString s = sIndentation + "PrimaryItem:\n";

  s += sSpacer + "Item Name = ";
  s += m_sItemName + "\n";

  s += sSpacer + "Item Id = ";
  sprintf( tmpBuf, "%ld", m_nItemId );
  s += tmpBuf;
  s += "\n";

  s += sSpacer + "Table Id = ";
  sprintf( tmpBuf, "%ld", m_nTableId );
  s += tmpBuf;
  s += "\n";

  s += sSpacer + "Section Mask = ";
  sprintf( tmpBuf, "%ld", m_nSectionMask );
  s += tmpBuf;
  s += "\n";

  s += sSpacer + "Transition Id = ";
  sprintf( tmpBuf, "%ld", m_nTransitionId );
  s += tmpBuf;
  s += "\n\n";

  s += sSpacer + "Item Number = ";
  sprintf( tmpBuf, "%ld", m_nItemNumber );
  s += tmpBuf;
  s += "\n";

  s += sSpacer + "Project Id = ";
  sprintf( tmpBuf, "%ld", m_nProjectId );
  s += tmpBuf;
  s += "\n";

  s += sSpacer + "State Id = ";
  sprintf( tmpBuf, "%ld", m_nStateId );
  s += tmpBuf;
  s += "\n";

  s += sSpacer + "Transition Id = ";
  sprintf( tmpBuf, "%ld", m_nTransitionId );
  s += tmpBuf;
  s += "\n";

  s += sSpacer + "ItemType = ";
  s += m_sItemType + "\n\n";

  // Spit out Attachment List
  TSAttachmentList::iterator itAttach = m_attachmentList.begin();
  for ( ; itAttach != m_attachmentList.end(); itAttach++ )
  {
    s += (*itAttach)->StringDump( sIndentation );
  }

  // Spit out Change History
  TSChangeList::iterator itChange = m_changeList.begin();
  for ( ; itChange != m_changeList.end(); itChange++ )
  {
    s += (*itChange)->StringDump( sIndentation );
  }

  // Spit out Display Fields
  TSDisplayFieldList::iterator itField = m_fieldList.begin();
  for ( ; itField != m_fieldList.end(); itField++ )
  {
    s += (*itField)->StringDump( sIndentation );
  }

  // Spit out Item Links
  TSItemLinkList::iterator itLink = m_linkList.begin();
  for ( ; itLink != m_linkList.end(); itLink++ )
  {
    s += (*itLink)->StringDump( sIndentation );
  }

  return s;
}
