// TSServer.cpp: implementation of the TSServer and TSSchema classes.
//
/////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <stdlib.h>
#include "TSAttachment.h"
#include "TSAuxiliaryItem.h"
#include "TSChangeHistory.h"
#include "TSDisplayField.h"
#include "TSItem.h"
#include "TSItemLink.h"
#include "TSField.h"
#include "TSPrimaryItem.h"
#include "TSProject.h"
#include "TSRecord.h"
#include "TSRecordRef.h"
#include "TSServer.h"
#include "TSTransition.h"
#include "TSUtility.h"



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 * Global Error status functions.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

static int TSErrorCode = TS_OK;

void TSSetLastError( int errorCode )
{
  TSErrorCode = errorCode;
}

int TSGetLastError()
{
  return TSErrorCode;
}

void TSClearLastError()
{
  TSSetLastError( TS_OK );
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Methods for the Schema class.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

TSSchema::~TSSchema()
{
  fieldList.EmptyAndDestroyList();
}

TSObject* TSSchema::NewObject()
{
  return new TSSchema;
}

TSObject* TSSchema::Duplicate( int /*type = 0*/ )
{
  TSSchema* obj = new TSSchema();
  obj->tableId = tableId;
  obj->name = name;
  obj->userDefinedFields = userDefinedFields;
  fieldList.Duplicate( &obj->fieldList );
  return obj;
}

// Copy the contents of sourceSchema into this.
void TSSchema::Copy( TSObject* sourceSchema )
{
  TSSchema* schema = static_cast<TSSchema*>( sourceSchema );
  tableId = schema->tableId;
  name = schema->name;
  schema->fieldList.Copy( &fieldList );
}

TSString TSSchema::StringDump( int      /*recursive*/,
                               TSString indentation )
{
  char tmp[24];
  TSString s = indentation;

  s += "tableId = ";
  sprintf( tmp, "%ld", tableId );
  s += tmp;
  s += "\n";
  s += indentation;
  s += "name = ";
  s += name;
  s += "\n";
  if ( userDefinedFields )
  {
    s += "Has User Defined Fields\n";
  }
  else
  {
    s += "Does Not Have User Defined Fields\n";
  }
  return s;
}

int TSSchema::SocketString( TSString& /*str*/ )
{
  // Not needed for a schema since it's a client-side-only object.
  TSSetLastError( TS_ERROR );
  return TSGetLastError();
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Methods for the Server class.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

TSServer::TSServer() :
  protocolString  ( "" ),
  serverName      ( "" ),
  directoryName   ( "" ),
  dllName         ( "" ),
  portNumber      ( 0 ),
  proxyString     ( "" ),
  authString      ( "" ),
  dllWebAddress   ( "" ),
  errorMsg        ( "" ),
  schemaCache     ( ),
  m_pSocket       ( NULL )
{
}

TSServer::~TSServer()
{
  schemaCache.EmptyAndDestroyList();
  if ( m_pSocket )
  {
    delete m_pSocket;
  }
}

int TSServer::Connect( const char* userName,
                       const char* password,
                       const char* serverAddress,
                       const char* proxyAddress /*=NULL*/ )
{
  int         len;
  char        delim;
  char*       host;
  char*       hostptr;
  const char* from;
  char        szBuffer[1024];

  TSClearLastError();
  // Some basic error checking
  if ( userName == NULL || *userName == '\0' )
  {
    errorMsg = "No username specified.";
    TSSetLastError( TS_INVALID_USER );
    return TSGetLastError();
  }
  if ( password == NULL )
  {
    errorMsg = "No password specified.";
    TSSetLastError( TS_INVALID_USER );
    return TSGetLastError();
  }
  if ( serverAddress == NULL || *serverAddress == '\0' )
  {
    errorMsg = "No server address specified.";
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  // Copy the server address, stripping of leading and trailing spaces
  // and changing backslashes to forward slashes, uppercase to lowercase
  host    = (char*)malloc( strlen( serverAddress ) + 1 );
  from    = serverAddress;
  hostptr = host;
  while ( *from && iswspace( *from ) ) from++;
  while ( *from && !iswspace( *from ) )
  {
    if ( *from == '\\' )
    {
      *hostptr++ = '/';
      from++;
    }
    else
    {
      *hostptr++ = (char)tolower( *from++ );
    }
  }
  *hostptr = '\0';
  hostptr = host;

  // Strip protocol
  if ( strncmp( hostptr, "http://", 7 ) == 0 )
  {
    protocolString = "http";
    portNumber     = 80;
    hostptr += 7;
  }
  else if ( strncmp( host, "https://", 8 ) == 0 )
  {
    protocolString = "https";
    portNumber     = 443;
    hostptr += 8;
  }
  else
  {
    protocolString = "http";
    portNumber     = 80;
  }

  // Strip server name
  len = strcspn( hostptr, ":/" );
  if ( len == 0 )
  {
    errorMsg = "No server address specified.";
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }
  delim = hostptr[len];
  hostptr[len] = '\0';
  serverName = hostptr;
  hostptr += len;
  if ( delim != '\0' )
  {
    hostptr++;
  }

  // Strip the port number
  if ( delim == ':' )
  {
    len = strcspn( hostptr, "/" );
    if ( len != 0 )
    {
      delim = hostptr[len];
      hostptr[len] = '\0';
      int nTmpPort = atol( hostptr );
      if ( nTmpPort > 0 )
      {
        portNumber = nTmpPort;
      }
      hostptr += len;
      if ( delim != '\0' )
      {
        hostptr++;
      }
    }
  }

  directoryName = "tmtrack";
  dllName       = "tmtrack.dll?";
  len = strlen( hostptr );
  if ( len > 0 )
  {
    // Get the directory and dll names
    char* cp;

    // If the last character is a question mark or slash, strip it off
    cp = &hostptr[ len - 1 ];
    if ( *cp == '?' || *cp == '/' )
    {
      len--;
      hostptr[ len ] = '\0';
    }

    // if a dll is specified
    if ( ( len > 4 ) && ( strcmp( &hostptr[ len - 4 ], ".dll"  ) == 0 ) )
    {
      // Strip off the dll name
      cp = strrchr( hostptr, '/' );
      if ( cp == NULL )
      {
        // No more slashes, rest of string must be the dll
        dllName = hostptr;
        directoryName = "";
      }
      else
      {
        dllName = &cp[1];
        *cp = '\0';
        // rest of string must be directory name
        directoryName = hostptr;
      }
      dllName += "?";
    }
    else
    {
      directoryName = hostptr;
    }
  }
    
  free( host );

  if ( proxyAddress != NULL )
  {
    proxyString = proxyAddress;
  }

  TSString authentication;
  authentication = userName;
  authentication += ":";
  authentication += password;
  authString = "Authorization: Basic ";
  authString += EncodePassword( authentication.GetBuffer());

  sprintf( szBuffer, "%s://%s:%d/%s/%s",
            protocolString.GetBuffer(),
            serverName.GetBuffer(),
            portNumber,
            directoryName.GetBuffer(),
            dllName.GetBuffer() );
  dllWebAddress = szBuffer;

  int nReturn = ValidateVersion();
  if ( nReturn != TS_OK )
  {
    // 0 is a version failure.
    if ( nReturn == 0 )
    {
      TSSetLastError( TS_INVALID_VERSION );
    }
    else // < 0 is another kind of error;
    {
      TSSetLastError( nReturn );
    }
    return TSGetLastError();
  }

  return TS_OK;
}

TSString TSServer::EncodePassword( const char* password )
{
  TSString out;
  unsigned char buf[3];
  int idx = 0;
	static char base64Table[] = 
		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
  while ( *password )
  {
    buf[idx++] = *password++;
    if ( idx == 3 )
    {
      // Output the 4 encoded bytes
      out += base64Table[buf[0] >> 2];                                   // first 6-bits of byte 0
      out += base64Table[((( buf[0] & 0x03 ) << 4 )) | ( buf[1] >> 4 )]; // last 2-bits of byte 0 + first 4-bits of byte 1
      out += base64Table[((( buf[1] & 0x0f ) << 2 )) | ( buf[2] >> 6 )]; // last 4-bits of byte 1 + first 2-bits of byte 2
      out += base64Table[buf[2] & 0x3f];                                 // last 6-bits of byte 2
      idx = 0;
    }
  }
  // Special case handling of last 1 or 2 characters
  if ( idx == 1 )
  {
    out += base64Table[buf[0] >> 2];               // first 6-bits of byte 0
    out += base64Table[(( buf[0] & 0x03 ) << 4 )]; // last 2-bits of byte 0
    out += "==";                                   // == pad
  }
  else if ( idx == 2 )
  {
    out += base64Table[buf[0] >> 2];                                   // first 6-bits of byte 0
    out += base64Table[((( buf[0] & 0x03 ) << 4 )) | ( buf[1] >> 4 )]; // last 2-bits of byte 0 + first 4-bits of byte 1
    out += base64Table[(( buf[1] & 0x0f ) << 2 )];                     // last 4-bits of byte 1
    out += "=";                                                        // = pad
  }
  return out;
}

const char* TSServer::GetDllWebAddress()
{
  return dllWebAddress.GetBuffer();
}

const char* TSServer::GetLastErrorMessage()
{
  return errorMsg.GetBuffer();
}

TSSocket* TSServer::OpenSocket()
{
  // Clear out the error message string before we get started
  errorMsg = "";
  TSClearLastError();

  TSSocket* socket = new TSSocket;
  if ( socket == NULL )
  {
    errorMsg = "Error allocating memory for socket.";
    TSSetLastError( TS_MEMORY_ERROR );
    return NULL;
  }
  if ( !socket->Create( protocolString.GetBuffer(),
                        directoryName.GetBuffer(),
                        dllName.GetBuffer(),
                        authString.GetBuffer(),
                        proxyString.GetBuffer() ) )
  {
    errorMsg = "Socket Create failed.";
    TSSetLastError( TS_SOCKET_CREATE_FAILED );
    delete socket;
    return NULL;
  }
  if ( !socket->Connect( serverName.GetBuffer(), portNumber ))
  {
    errorMsg = "Socket Connect failed.";
    TSSetLastError( TS_SOCKET_CONNECT_FAILED );
    delete socket;
    return NULL;
  }

  return socket;
}

TSSocket* TSServer::GetSocket()
{
  if ( m_pSocket == NULL )
  {
    m_pSocket = OpenSocket();
    if ( m_pSocket == NULL )
    {
      return NULL;
    }
  }

  if ( alternateUser.Length() > 0 )
  { // Set the alternate user in the socket
    m_pSocket->SetAlternateUser( alternateUser );
  }

  return m_pSocket;
}

int TSServer::Send( TSSocket*   socket,
                    const char* str,
                    int len /*=0*/ )
{
  TSClearLastError();
  // Send can only return 0, 1 or 2 due to the way the methods
  //  that call it are checking Send's return value ... if ( !Send() ). 
  //  0 == error, 1 == success with error message set, 2 == success.
  int code = -1;
  int nTimes = 0;
  char errmsg[80];
  
  if ( len == 0 )
  {
    len = strlen( str );
  }

  while ( code != 200 )
  {
    if ( socket->Send( str, len ) == SOCKET_ERROR )
    {
      // If we are using the keep-alive socket and we failed,
      // we probably just timed-out. Try to open a new socket.
      if ( (socket == m_pSocket) && (nTimes == 0) )
      {
        delete socket;
        m_pSocket = NULL;
        socket = GetSocket();
        if ( socket == NULL )
        {
          return 0; // error set in Open(Get)Socket
        }
        nTimes++;
        continue;
      }
      TSSetLastError( TS_SOCKET_WRITE_ERROR );
      return 0;
    }

    if ( !socket->GetHttpStatusCode( code ) ||
         ( code != 200 && code != 400 ) ||
         ( nTimes++ > 10 ) )
    {
      TSSetLastError( TS_SERVER_ERROR );
      sprintf( errmsg, "Server Error: %d", code );
      errorMsg = errmsg;
      return 0;
    }

    // Now this is some ugly code. Let me try to explain why I did
    // this. First of all, the only time I think we will get a 400
    // error is when we have previously gotten a 12045 error when
    // trying to send an http request. This error occurs when the
    // certificate authority is unknown when connecting through
    // SSL (https). When this occurs, the only way I was able to
    // get it to work was to close down the socket connection and
    // re-open it. When re-opened, it appears to work correctly.
    // Don't ask me why, I don't know. I just happened upon this
    // as a solution. See BUG33627
    if ( code == 400 )
    {
      socket->Close();
      socket->Connect( serverName.GetBuffer(), portNumber );
    }
  }
  
  // The stuff coming across the socket looks like:
  //    retCode, "errorMsg", result (s).
  //  Send pulls out the retCode and errorMsg, and leaves
  //    the result(s) for the function that called Send.
  //  RetCode will equal either 1 (DB_ERRWITHRESULTS), or
  //    2 (DB_SUCCESS). This is set in CAppDbServer::SendErrors
  //    If errorcount is > 0 it is set to 1, else it is set to 2.
  //    If retCode equals 1, then errorMsg has also been set. The 
  //    message can be retrieved with GetLastErrorMessage().
  //  The FIRST result in result(s) is also set in CAppDbServer::SendErrors.
  //    It is equal to SendErrors' second parameter and is typically
  //    set equal to the result of whatever function was called,
  //    e.g., result = pRec->AddRecord(...).
  //  Any further results are set using other send methods, like
  //    pRec->Send( pStream ) or list.Send( pStream ).

  int retCode;
  int nReturn = socket->ReceiveInt( &retCode );// retCode will be a 1 or 2
  if ( retCode == 0 )
  {
    nReturn = socket->ReceiveString( &errorMsg );
    if ( nReturn != TS_OK )
    {
      // Set the error message since we did not get one from the server.
      errorMsg = "Socket error from initial Send() return code.\n";
      errorMsg += "One reason might be your database needs upgrading.\n";
      errorMsg += "Check event viewer if possible.\n";
    }
    TSSetLastError( TS_SOCKET_READ_ERROR );
    return 0; // Don't return TSGetLastError() here.
  }
  if ( nReturn != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    sprintf( errmsg, "Failure to receive return code: %d", nReturn );
    errorMsg = errmsg;
    return 0;
  }
  nReturn = socket->ReceiveString( &errorMsg );
  if ( retCode == 1 && nReturn != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    sprintf( errmsg, "Failure to receive return message: %d", nReturn );
    errorMsg = errmsg;
    return 0;
  }
  return retCode; // Either a 1 or a 2
}

/********************************************************************
  DATA ACCESS FUNCTIONS
 ********************************************************************/

int TSServer::AddField( TSRecord* field, int tableId, int fieldType, int& fieldId )
{
  //  This function will add a record to the FIELDS table.  The
  //    tableId parameter will be used to set the TS_TABLEID field
  //    in the new record.  There is no need to set this value before
  //    passing it into this function.
  //  The return value is a TS_xxx error code.
  //  NOTE:  A return of TS_OK does not necessarily indicate a success
  //    ALWAYS check the fieldId.
  //  The fieldId will be 0 if failed, otherwise it's the TS_ID
  //    for the new FIELDS table record.

  if ( !field )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  // Set the tableid in the record
  if ( field->SetInt( "tableid", tableId ) != TS_OK )
  {
    // TSErrorCode is set in SetInt()
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=AddField2&Params=";
  TSEncodeInt( fieldType, params );
  s += params;
  field->SocketString( params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    fieldId = 0;
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  fieldId = nResult;
  return TS_OK;
}

int TSServer::AddRecord( TSRecord* rec,
                         TSRecord* newRecord /*=NULL*/ )
{
  if ( !rec )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=AddRecord&Params=";
  TSEncodeInt( rec->tableId, params );
  s += params;
  TSEncodeInt( rec->fieldType, params );
  s += params;
  rec->SocketString( params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( newRecord != NULL )
  {
    if ( newRecord->Receive( m_pSocket ) != TS_OK )
    {
      // TSErrorCode is set in Receive()
      return TSGetLastError();
    }
  }
  else
  {
    TSRecord dummy( rec->tableId, this );
    dummy.fieldType = rec->fieldType;
    if ( dummy.Receive( m_pSocket ) != TS_OK )
    {
      // TSErrorCode is set in Receive()
      return TSGetLastError();
    }
  }

  return TS_OK;
}

int TSServer::AddTable( TSRecord* table,
                        TSRecord* solution /*=NULL*/,
                        int optionalFields /*=0*/,
                        int reserved /*=0*/ )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSRecord localSolution( TS_TBLID_SOLUTIONS, this );

  TSString s;
  TSString params;
  s = "Func=AddTable&Params=";
  table->SocketString( params );
  s += params;
  if ( solution != NULL )
  {
    solution->SocketString( params );
  }
  else
  {
    localSolution.SocketString( params );
  }
  s += params;
  TSEncodeInt( optionalFields, params );
  s += params;
  TSEncodeInt( reserved, params );
  s += params;
  
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( nResult <= 0 )
  {
    if ( nResult == 0 ) nResult = TS_ERROR;
    TSSetLastError( nResult );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::BuildFieldList( int          tableId,
                              TSFieldList& fieldList )
{
  TSSchema* schema = GetSchema( tableId );
  if ( schema == NULL )
  {
    // TSErrorCode is set in GetSchema()
    return TSGetLastError();
  }

  if ( schema->fieldList.Duplicate( &fieldList ) != TS_OK )
  {
    // TSErrorCode is set in Duplicate()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ClearAlternateUser()
{
  // We need to release any license that the alternate user may be consuming
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ClearAlternateUser";

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  // Empty our version of the alternateUser so it is no longer sent
  // Note: do not do this until after the Send call
  alternateUser.Empty();
  m_pSocket->ClearAlternateUser();

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  // nResult will be the error code, TS_OK on success
  if ( nResult != TS_OK )
  {
    TSSetLastError( nResult );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::DeleteRecord( int tableId,
                            int id )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=DeleteRecord&Params=";
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeInt( id, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }
  
  return TS_OK;
}

int TSServer::ExecuteSQL( TSString& sql )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ExecuteSQL&Params=";
  TSEncodeString( sql, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }
 
  return TS_OK;
}

int TSServer::GetConnectionInfo( char** dsn,
                                 char** databaseName,
                                 char** serverName )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  s = "Func=GetConnectionInfo&Params=";

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  TSString tmpStr;
  if ( m_pSocket->ReceiveString( &tmpStr ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    return TSGetLastError();
  }

  *dsn = new char[tmpStr.Length()+1];
  if ( *dsn == NULL )
  {
    TSSetLastError( TS_MEMORY_ERROR );
    return TSGetLastError();
  }
  strcpy( *dsn, tmpStr.GetBuffer() );

  tmpStr = "";
  if ( m_pSocket->ReceiveString( &tmpStr ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    return TSGetLastError();
  }

  *databaseName = new char[tmpStr.Length()+1];
  if ( *databaseName == NULL )
  {
    TSSetLastError( TS_MEMORY_ERROR );
    return TSGetLastError();
  }
  strcpy( *databaseName, tmpStr.GetBuffer() );

  tmpStr = "";
  if ( m_pSocket->ReceiveString( &tmpStr ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    return TSGetLastError();
  }

  *serverName = new char[tmpStr.Length()+1];
  if ( *serverName == NULL )
  {
    TSSetLastError( TS_MEMORY_ERROR );
    return TSGetLastError();
  }
  strcpy( *serverName, tmpStr.GetBuffer() );

  return TS_OK;
}

int TSServer::GetConnectionInfo( char* dsn,
                                 char* databaseName,
                                 char* serverName )
{
  if ( !dsn || ! databaseName || !serverName )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  s = "Func=GetConnectionInfo&Params=";

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  TSString tmpStr;
  if ( m_pSocket->ReceiveString( &tmpStr ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    return TSGetLastError();
  }

  strcpy( dsn, tmpStr.GetBuffer() );
  tmpStr = "";

  if ( m_pSocket->ReceiveString( &tmpStr ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    return TSGetLastError();
  }

  strcpy( databaseName, tmpStr.GetBuffer() );
  tmpStr = "";

  if ( m_pSocket->ReceiveString( &tmpStr ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    return TSGetLastError();
  }

  strcpy( serverName, tmpStr.GetBuffer() );

  return TS_OK;
}

int TSServer::GetDbInfo( int   infoType,
                         void* out )
{
  if ( !out )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  if ( infoType == TS_DBINFO_EXPIRATIONDATE )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=GetDbInfo&Params=";
  TSEncodeInt( infoType, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  int nReturn = 0;
  switch ( infoType )
  {
    case TS_DBINFO_VERSION:
    case TS_DBINFO_ROOTPROJECTID:
    case TS_DBINFO_ROOTFOLDERID:
    case TS_DBINFO_OBJECTVERSION:
    case TS_DBINFO_DBMSVER:
    case TS_DBINFO_VARCHARLENGTH:
    case TS_DBINFO_LONGVARCHARLENGTH:
    case TS_DBINFO_ALLOWANONYMOUS:
    {
      nReturn = m_pSocket->ReceiveInt( static_cast<int*>( out ));
      break;
    }
  
    case TS_DBINFO_EXITURL:
    case TS_DBINFO_ROOTPROJECTNAME:
    case TS_DBINFO_DBMSNAME:
    {
      *( static_cast<TSString*>( out )) = "";
      nReturn = m_pSocket->ReceiveString( static_cast<TSString*>( out ));
      break;
    }

    default:
      break;
  }

  if ( nReturn != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt() and ReceiveString()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::GetInt( int         tableId,
                      const char* columnName,
                      int         recId,
                      int*        pInt )
{
  if ( !columnName || !pInt )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=GetInt&Params=";
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeString( columnName, params );
  s += params;
  TSEncodeInt( recId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    *pInt = 0;
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    *pInt = 0;
    return TSGetLastError();
  }

  *pInt = nResult;
  return TS_OK;
}

TSSchema* TSServer::GetSchema( int tableId )
{
  TSClearLastError();
  TSSchema* schema;
  TSPosition* pos = schemaCache.GetFirst();
  while ( pos )
  {
    schema = static_cast<TSSchema*>( schemaCache.GetAt( pos ));
    if ( schema->tableId == tableId )
    {
      return schema;
    }
    pos = schemaCache.GetNext(pos);
  }

  // Must not have found the schema for this record type.
  // Send a message to the server to retrieve it.
  TSSocket* socket = OpenSocket();
  if ( socket == NULL )
  {
    // TSErrorCode is set in Open(Get)Socket
    return NULL;
  }

  TSString s;
  TSString params;
  s = "Func=ReadSchema&Params=";
  TSEncodeInt( tableId, params );
  s += params;

  if ( !Send( socket, s ))
  {
    // TSErrorCode is set in Send()
    delete socket;
    return NULL;
  }

  int nResult;
  if ( socket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    delete socket;
    return NULL;
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    delete socket;
    return NULL;
  }

  // Receive the id, name and count.
  int count;
  int id;
  TSString name;
  schema = new TSSchema;
  schema->tableId = tableId;
  if ( socket->ReceiveInt( &id ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    delete socket;
    return NULL;
  }

  if ( socket->ReceiveString( &schema->name ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    delete socket;
    return NULL;
  }

  if ( socket->ReceiveInt( &schema->userDefinedFields ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    delete socket;
    return NULL;
  }

  if ( socket->ReceiveInt( &count ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    delete socket;
    return NULL;
  }

  TSField* field;
  for ( int ii = 0; ii < count; ii++ )
  {
    field = new TSField();
    if ( field->ReceiveSchema( socket ) != TS_OK )
    {
      // TSErrorCode is set in ReceiveSchema()
      delete socket;
      return NULL;
    }

    if ( schema->fieldList.AddTail( field ) != TS_OK )
    {
      // TSErrorCode is set in AddTail()
      delete socket;
      return NULL;
    }
  }

  if ( schemaCache.AddHead( schema ) != TS_OK )
  {
    // TSErrorCode is set in AddHead()
    delete socket;
    return NULL;
  }

  delete socket;
  return schema;
}

int TSServer::GetPrivilegeName( enum privId_e privId,
                                char**        privName )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=GetPrivilegeName&Params=";
  TSEncodeInt( privId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  TSString tmpStr;
  if ( m_pSocket->ReceiveString( &tmpStr ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    return TSGetLastError();
  }

  if ( tmpStr.Length() > 0 )
  {
    *privName = new char[tmpStr.Length() + 1];
    if ( *privName == NULL )
    {
      TSSetLastError( TS_MEMORY_ERROR );
      return TSGetLastError();
    }
    strcpy( *privName, tmpStr.GetBuffer() );
  }
  else
  {
    *privName = NULL;
  }

  return TS_OK;
}

int TSServer::GetProjectList( int nTableId,
                              int nProjectsMask,
                              TSProjectList& projectList )
{
  TSProject*  pProject;

  while ( projectList.size() > 0 )
  {
    pProject = projectList.front();
    projectList.pop_front();
    delete pProject;
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSGetProjectList&Params=";
  TSEncodeInt( nTableId, params );
  s += params;
  TSEncodeInt( nProjectsMask, params );
  s += params;
  
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( nResult != TS_OK )
  {
    TSSetLastError( nResult );
    return TSGetLastError();
  }

  int nErrCode = TS_OK;
  while ( nErrCode == TS_OK )
  {
    TSProject* pProject = new TSProject( *this );
    nErrCode = pProject->Receive( m_pSocket );
    if ( nErrCode != TS_OK )
    {
      delete pProject;
      break;
    }
    projectList.push_back( pProject );
  }
  
  return TS_OK;
}

int TSServer::GetString( int         tableId,
                         const char* columnName,
                         int         recId,
                         char**      string )
{
  if ( !columnName )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=GetString&Params=";
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeString( columnName, params );
  s += params;
  TSEncodeInt( recId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  TSString tmpStr;
  if ( m_pSocket->ReceiveString( &tmpStr ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveString()
    return TSGetLastError();
  }

  if ( tmpStr.Length() > 0 )
  {
    *string = new char[tmpStr.Length() + 1];
    if ( *string == NULL )
    {
      TSSetLastError( TS_MEMORY_ERROR );
      return TSGetLastError();
    }
    strcpy( *string, tmpStr.GetBuffer() );
  }
  else
  {
    *string = NULL;
  }

  return TS_OK;
}

int TSServer::GetString( int         tableId,
                         const char* columnName,
                         int         recId,
                         char*       string,
                         int         size )
{
  if ( !columnName || !string )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  char* tmp;
  if ( GetString( tableId, columnName, recId, &tmp ) != TS_OK )
  {
    delete [] tmp;
    tmp = 0;
    // TSErrorCode is set in GetString()
    return TSGetLastError();
  }

  strncpy( string, tmp, size );
  
  delete [] tmp;
  tmp = 0;
  return TS_OK;
}

int TSServer::GetSubmitTransition( int  projectId,
                                   int* id )
{
  if ( !id )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=GetSubmitTransition&Params=";
  TSEncodeInt( projectId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  if ( m_pSocket->ReceiveInt( id ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::GetTableIdByDatabaseNameEx( const char* tableDbName, 
                                          int*        tableId )
{
  if ( !tableDbName )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=GetTableIdByDatabaseNameEx&Params=";
  TSEncodeString( tableDbName, params );
  s += params;

  if ( !Send( m_pSocket, s ) )
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    *tableId = 0;
    return TSGetLastError();
  }

  *tableId = nResult;
  return TS_OK;
}

int TSServer::HasPrivilege( int userId,
                            int projectId,
                            int maskNumber,
                            int mask )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=HasPrivilege&Params=";
  TSEncodeInt( userId, params );
  s += params;
  TSEncodeInt( projectId, params );
  s += params;
  TSEncodeInt( maskNumber, params );
  s += params;
  TSEncodeInt( mask, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::HasGroupPrivilege( int           groupId,
                                 int           itemId,
                                 enum privId_e privId )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=HasGroupPrivilege&Params=";
  TSEncodeInt( groupId, params );
  s += params;
  TSEncodeInt( itemId, params );
  s += params;
  TSEncodeInt( privId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::HasUserPrivilege( int           userId,
                                int           itemId,
                                enum privId_e privId )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=HasUserPrivilege&Params=";
  TSEncodeInt( userId, params );
  s += params;
  TSEncodeInt( itemId, params );
  s += params;
  TSEncodeInt( privId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::HasRecordPrivilege( enum recPriv_e recPriv,
                                  int  userId,
                                  int  tableId,
                                  int  recId )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=HasRecordPrivilege&Params=";
  TSEncodeInt( recPriv, params );
  s += params;
  TSEncodeInt( userId, params );
  s += params;
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeInt( recId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::HasValidLicense( const char* solutionIdent )
{
  if ( !solutionIdent )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=HasValidLicense&Params=";
  TSEncodeString( solutionIdent, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::MoveFolder( int folderId,
                          int newParentId,
                          int position )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=MoveFolder&Params=";
  TSEncodeInt( folderId, params );
  s += params;
  TSEncodeInt( newParentId, params );
  s += params;
  TSEncodeInt( position, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::MoveProject( int projectId,
                           int newParentId,
                           int position )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=MoveProject&Params=";
  TSEncodeInt( projectId, params );
  s += params;
  TSEncodeInt( newParentId, params );
  s += params;
  TSEncodeInt( position, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadAllRecords( TSRecordList* list,
                              int           tableId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadAllRecords&Params=";
  TSEncodeInt( tableId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }
    
  return TS_OK;
}

int TSServer::ReadAttachmentList( TSRecordList* list,
                                  int           tableId,
                                  int           recId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadAttachmentList&Params=";
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeInt( recId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadAvailableTransitionList( TSRecordList* list,
                                           int           tableId,
                                           int           recId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadAvailableTransitionList&Params=";
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeInt( recId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadChangeList( TSRecordList* list,
                              int           caseId,
                              BOOL          newFirst /*=FALSE*/ )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadChangeList&Params=";
  TSEncodeInt( caseId, params );
  s += params;
  TSEncodeInt( newFirst, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadFieldsByProject( TSRecordList* list, int projectId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadFieldsByProject&Params=";
  TSEncodeInt( projectId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadFolderItems( TSRecordList* list,
                               int           folderId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadFolderItems&Params=";
  TSEncodeInt( folderId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadFolderList( TSRecordList* list,
                              int           owner,
                              int           parentId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadFolderList&Params=";
  TSEncodeInt( owner, params );
  s += params;
  TSEncodeInt( parentId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadProjectSelectionList( TSRecordList* list,
                                        int           selectid )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadProjectSelectionList&Params=";
  TSEncodeInt( selectid, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadProjectTransitionList( TSRecordList* list,
                                         int           transId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadProjectTransitionList&Params=";
  TSEncodeInt( transId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadPropertyList( TSRecordList* list,
                                int           transId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadPropertyList&Params=";
  TSEncodeInt( transId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }
  
  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadRecord( TSRecord* rec,
                          int       id )
{
  if ( !rec )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadRecord&Params=";
  TSEncodeInt( rec->tableId, params );
  s += params;
  TSEncodeInt( rec->fieldType, params );
  s += params;
  TSEncodeInt( id, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  // Only ignore non-db fields if we are reading the fields table
  // Server side only sends db & core fields
  if ( rec->Receive( m_pSocket, ( rec->tableId == TS_TBLID_FIELDS && rec->fieldType == 0 ) ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadRecordByItemNumber( TSRecord* pRec,
                                      TSString  sItemNumber,
                                      int       nProjectId )
{
  if ( !pRec )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadRecordByItemNumber&Params=";
  TSEncodeString( sItemNumber, params );
  s += params;
  TSEncodeInt( nProjectId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( pRec->Receive( m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadRecordForId( TSRecord*   record,
                               const char* searchId )
{
  if ( !record || !searchId )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadRecordForItemId&Params=";
  TSEncodeInt( record->tableId, params );
  s += params;
  TSEncodeInt( record->fieldType, params );
  s += params;
  TSEncodeString( searchId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }
  
  // Only ignore non-db fields if we are reading the fields table
  // Server side only sends db & core fields
  if ( record->Receive( m_pSocket, ( record->tableId == TS_TBLID_FIELDS && record->fieldType == 0 ) ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadRecordListWithWhere( TSRecordList* list,
                                       int           tableId,
                                       const char*   whereClause )
{
  if ( !list || !whereClause )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadListWithWhere&Params=";
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeString( whereClause, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

// Reads a record, filtering by the where clause and returning only the desired fields.
int TSServer::ReadRecordListWithWhere( TSRecordList*           list, 
                                       int                     tableId, 
                                       const char*             whereClause,
                                       const std::vector<int>& fields )
{
  if ( !list || !whereClause )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadListSelectedFieldsWithWhere&Params=";
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeString( whereClause, params );
  s += params;
  // Encode the vector of selected fields.
  TSEncodeInt( fields.size(), params );
  s += params;
  std::vector<int>::const_iterator it = fields.begin();
  while ( it != fields.end() )
  {
    TSEncodeInt( *it, params );
    s += params;
    it++;
  }

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadRecordWithWhere( TSRecord*   record,
                                   const char* whereClause )
{
  if ( !record || !whereClause )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadRecordWithWhere&Params=";
  TSEncodeInt( record->tableId, params );
  s += params;
  TSEncodeInt( record->fieldType, params );
  s += params;
  TSEncodeString( whereClause, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  // Only ignore non-db fields if we are reading the fields table
  // Server side only sends db & core fields
  if ( record->Receive( m_pSocket, ( record->tableId == TS_TBLID_FIELDS && record->fieldType == 0 ) ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

// Reads a record, filtering by the where clause and returning only the desired fields.
int TSServer::ReadRecordWithWhere( TSRecord*               record,
                                   const char*             whereClause,
                                   const std::vector<int>& fields )
{
  if ( !record || ! whereClause )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadSelectedFieldsWithWhere&Params=";
  TSEncodeInt( record->tableId, params );
  s += params;
  TSEncodeInt( record->fieldType, params ); // This is the type.
  s += params;
  TSEncodeString( whereClause, params );
  s += params;
  TSEncodeInt( fields.size(), params );     // Encode the vector of selected fields.
  s += params;
  std::vector<int>::const_iterator it = fields.begin();
  while ( it != fields.end() )
  {
    TSEncodeInt( *it, params );
    s += params;
    it++;
  }

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  // Only ignore non-db fields if we are reading the fields table
  // Server side only sends db & core fields
  if ( record->Receive( m_pSocket, ( record->tableId == TS_TBLID_FIELDS && record->fieldType == 0 ) ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadReport( TSRecord*   report,
                          const char* name )
{
  if ( !report || !name )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  TSString modifiedName;

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadReport&Params=";
  TSEncodeInt( TS_TBLID_REPORTS, params );
  s += params;
  TSEncodeInt( 0, params ); // reportType, currently needs to be 0.
  s += params;
  TSEncodeString( name, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( report->Receive( m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadReportList( TSRecordList* list,
                              long          userId,
                              int           perm )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadReportList&Params=";
  TSEncodeInt( userId, params );
  s += params;
  TSEncodeInt( perm, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadSelectionList( TSRecordList* list,
                                 int           fieldId,
                                 int           projectId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadSelectionList2&Params=";
  TSEncodeInt( fieldId, params );
  s += params;
  TSEncodeInt( projectId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadStateList( TSRecordList* list,
                             int           workflowId,
                             BOOL          incParent /*=FALSE*/ )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadStateList&Params=";
  TSEncodeInt( workflowId, params );
  s += params;
  TSEncodeInt( incParent, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadTransitionList( TSRecordList* list,
                                  int           projectId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadTransitionListEx&Params=";
  TSEncodeInt( projectId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadUserDefinedFields( TSRecordList* list,
                                     int           tableId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  TSSocket* socket = OpenSocket();
  if ( socket == NULL )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadUserDefinedFields&Params=";
  TSEncodeInt( tableId, params );
  s += params;

  if ( !Send( socket, s ))
  {
    // TSErrorCode is set in Send()
    delete socket;
    return TSGetLastError();
  }

  int nResult;
  if ( socket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    delete socket;
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    delete socket;
    return TSGetLastError();
  }

  if ( list->Receive( this, socket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    delete socket;
    return TSGetLastError();
  }

  delete socket;
  return TS_OK;
}

int TSServer::ReadUserListForPrivilege( TSRecordList* list,
                                        int           userid,
                                        TSRecordList* fieldList,
                                        int           mask )
{
  if ( !list || !fieldList )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadUserListForPrivilege&Params=";
  TSEncodeInt( userid, params );
  s += params;
  fieldList->SocketString( params );
  s += params;
  TSEncodeInt( mask, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadUserSelectionList( TSRecordList* list,
                                     int           fieldId,
                                     int           projectId, 
                                     BOOL          incDeleted /*=FALSE*/ )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadUserSelectionList&Params=";
  TSEncodeInt( fieldId, params );
  s += params;
  TSEncodeInt( projectId, params );
  s += params;
  TSEncodeInt( incDeleted, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadVCActions( TSRecordList* list,
                             int           caseId )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadVCActions&Params=";
  TSEncodeInt( caseId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReadVCActionsForModule( TSRecordList* list,
                                      const char*   filename,
                                      int           userid,
                                      int           action2 )
{
  if ( !list || !filename )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReadVCActionsForModule&Params=";
  TSEncodeString( filename, params );
  s += params;
  TSEncodeInt( userid, params );
  s += params;
  TSEncodeInt( action2, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::RefreshCache( int tableId )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s, params;
  s = "Func=RefreshCache&Params=";
  TSEncodeInt( tableId, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReleaseRecordLock( int nTableId, int nItemId,
                             int nProjectId /* = 0 */, int nItemNumber /* = 0 */ )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ReleaseRecordLock&Params=";
  TSEncodeInt( nTableId, params );
  s += params;
  TSEncodeInt( nItemId, params );
  s += params;
  TSEncodeInt( nProjectId, params );
  s += params;
  TSEncodeInt( nItemNumber, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( nResult != TS_OK )
  {
    TSSetLastError( nResult );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::ReleaseRecordLockById( int nTableId, int nItemId )
{
  return ReleaseRecordLock( nTableId, nItemId );
}

int TSServer::ReleaseRecordLockByNumber( int nProjectId, int nItemNumber )
{
  return ReleaseRecordLock( 0, 0, nProjectId, nItemNumber );
}

int TSServer::SetAlternateUser( const char* loginid,
                                const char* pwd,
                                bool        bValidatePwd /* = true */ )
{
  if ( !loginid )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=SetAlternateUser&Params=";

  TSString userName( loginid );

  if ( bValidatePwd )
  {
    userName += ":";
    userName += pwd;
  }

  userName = EncodePassword( userName );

  TSEncodeString( userName, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  // nResult will be the error code, TS_OK on success
  if ( nResult != TS_OK )
  {
    TSSetLastError( nResult );
    return TSGetLastError();
  }

  alternateUser = userName;

  return TS_OK;
}

int TSServer::SetExitUrl( const char* exiturl )
{
  if ( !exiturl )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=SetExitUrl&Params=";
  TSEncodeString( exiturl, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::SetGroupPrivilege( int           groupId,
                                 int           itemId,
                                 enum privId_e privId,
                                 bool          bGrant )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=SetGroupPrivilege&Params=";
  TSEncodeInt( groupId, params );
  s += params;
  TSEncodeInt( itemId, params );
  s += params;
  TSEncodeInt( privId, params );
  s += params;
  TSEncodeInt( bGrant, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::SetRecordLock( int nTableId, int nItemId,
                             bool bStealLockFlag,
                             int nProjectId /* = 0 */, int nItemNumber /* = 0 */ )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=SetRecordLock&Params=";
  TSEncodeInt( nTableId, params );
  s += params;
  TSEncodeInt( nItemId, params );
  s += params;
  TSEncodeInt( nProjectId, params );
  s += params;
  TSEncodeInt( nItemNumber, params );
  s += params;
  TSEncodeInt( bStealLockFlag, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( nResult != TS_OK )
  {
    TSSetLastError( nResult );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::SetRecordLockById( int nTableId, int nItemId,
                                 bool bStealLockFlag /* = false */ )
{
  return SetRecordLock( nTableId, nItemId, bStealLockFlag );
}

int TSServer::SetRecordLockByNumber( int nProjectId, int nItemNumber,
                                     bool bStealLockFlag /* = false */ )
{
  return SetRecordLock( 0, 0, bStealLockFlag, nProjectId, nItemNumber );
}

int TSServer::SetUserPrivilege( int           userId,
                                int           itemId,
                                enum privId_e privId,
                                bool          bGrant )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=SetUserPrivilege&Params=";
  TSEncodeInt( userId, params );
  s += params;
  TSEncodeInt( itemId, params );
  s += params;
  TSEncodeInt( privId, params );
  s += params;
  TSEncodeInt( bGrant, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::Submit( int*      nIssueId,
                      TSString& sLoginid,
                      TSRecord* pRec,
                      int       nTableId,
                      int       nProjectId,
                      int       nFolderId,
                      int       /*nType*/ )
{
  if ( !nIssueId || !pRec )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  return TSServer::Submit( nIssueId, sLoginid, pRec, nTableId, nProjectId, nFolderId );
}

// This version of submit returns the issue id as an int.
int TSServer::Submit( int*      nIssueId,
                      TSString& sLoginid,
                      TSRecord* pRec,
                      int       nTableId,
                      int       nProjectId,
                      int       nFolderId )
{
  if ( !nIssueId || !pRec )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=Submit&Params=";
  TSEncodeString( sLoginid, params );
  s += params;
  TSEncodeInt( nProjectId, params );
  s += params;
  TSEncodeInt( nTableId, params );
  s += params;
  TSEncodeInt( nFolderId, params );
  s += params;
  pRec->SocketString( params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  // Receive the issue id
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  *nIssueId = nResult;

  return TS_OK;
}

// This version of submit returns both the issue id as an int, and the
// ts_id as an int. This is probably the one you want to use if you will
// need the ts_id.
int TSServer::Submit( int*      nId,
                      int*      nIssueId,
                      TSString& sLoginid,
                      TSRecord* pRec,
                      int       nTableId,
                      int       nProjectId,
                      int       nFolderId )
{
  if ( !nId || !nIssueId || !pRec )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=SubmitReturnId&Params=";
  TSEncodeString( sLoginid, params );
  s += params;
  TSEncodeInt( nProjectId, params );
  s += params;
  TSEncodeInt( nTableId, params );
  s += params;
  TSEncodeInt( nFolderId, params );
  s += params;
  pRec->SocketString( params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  // First receive the issue id
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }
  *nIssueId = nResult;

  // Next, receive the ts_id
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }
  *nId = nResult;

  return TS_OK;
}

int TSServer::Transition( TSString& sLoginid,
                          TSRecord* pRec,
                          int       nProjectId,
                          int       nTableId,
                          int       nRecordId,
                          int       nTransition )
{
  if ( !pRec )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=Transition&Params=";
  TSEncodeString( sLoginid, params );
  s += params;
  TSEncodeInt( nProjectId, params );
  s += params;
  TSEncodeInt( nTableId, params );
  s += params;
  TSEncodeInt( nRecordId, params );
  s += params;
  TSEncodeInt( nTransition, params );
  s += params;
  pRec->SocketString( params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return nResult;
}

int TSServer::UpdateList( TSRecordList* list,
                          int           tableId,
                          int           recid1,
                          int           recid2 )
{
  if ( !list )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=UpdateList&Params=";
  TSEncodeInt( tableId, params );
  s += params;
  TSEncodeInt( recid1, params );
  s += params;
  TSEncodeInt( recid2, params );
  s += params;
  list->SocketString( params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( list->Receive( this, m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::UpdateRecord( TSRecord* rec,
                            TSRecord* newRecord /*=NULL*/ )
{
  if ( !rec )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=UpdateRecord&Params=";
  TSEncodeInt( rec->tableId, params );
  s += params;
  TSEncodeInt( rec->fieldType, params );
  s += params;
  rec->SocketString( params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  if ( newRecord != NULL )
  {
    if ( newRecord->Receive( m_pSocket ) != TS_OK )
    {
      // TSErrorCode is set in Receive()
      return TSGetLastError();
    }
  }
  else
  {
    TSRecord dummy( rec->tableId, this );
    dummy.fieldType = rec->fieldType;
    if ( dummy.Receive( m_pSocket ) != TS_OK )
    {
      // TSErrorCode is set in Receive()
      return TSGetLastError();
    }
  }

  return TS_OK;
}

int TSServer::ValidateUser( const char* loginid,
                            const char* pwd,
                            BOOL        validatePwd /*=TRUE*/,
                            int*        userId /*=NULL*/ )
{
  if ( !loginid )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ValidateUser&Params=";
  TSEncodeString( loginid, params );
  s += params;
  TSEncodeString( pwd, params );
  s += params;
  TSEncodeInt( validatePwd, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    if ( userId != NULL )
    {
      *userId = 0;
    }
    TSSetLastError( TS_INVALID_USER );
    return TSGetLastError();
  }
  else
  {
    if ( userId != NULL )
    {
      *userId = nResult;
    }
  }

  return TS_OK;
}

int TSServer::ValidateVersion()
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=ValidateAPIVersion&params=";
  TSEncodeInt( TS_APIVERSION, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

/////////////////////////////////////////////////////////////////////////////////
// Methods pertaining to Display Data Classes
/////////////////////////////////////////////////////////////////////////////////

int TSServer::ReceiveTSItem( TSSocket* socket,
                             TSItem&   item )
{
  if ( !socket )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  TSRecordRef& recordRef = item;
  if ( recordRef.Receive( socket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    TSSetLastError( TS_SOCKET_READ_ERROR );
    return TSGetLastError();
  }
  
  if ( socket->ReceiveInt( &item.m_nSectionMask ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( socket->ReceiveInt( &item.m_nTransitionId ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  int nErrCode = TS_OK;
  while ( nErrCode == TS_OK )
  {
    TSAttachment* pAttachment = new TSAttachment( *this );
    nErrCode = pAttachment->Receive( socket );
    if ( nErrCode != TS_OK )
    {
      delete pAttachment;
      break;
    }
    item.m_attachmentList.push_back( pAttachment );
  }
  // There's always a \r\n at the end of the list, but usually the \r has
  // already been read so read up till the next \n.
  char ch;
  while (( ch = socket->ReadChar()) != 0 )
  {
    if ( ch == '\n' )
    {
      break;
    }
  }

  nErrCode = TS_OK;
  while ( nErrCode == TS_OK )
  {
    TSChangeHistory* pChangeHistory = new TSChangeHistory( *this );
    nErrCode = pChangeHistory->Receive( socket );
    if ( nErrCode != TS_OK )
    {
      delete pChangeHistory;
      break;
    }
    item.m_changeList.push_back( pChangeHistory );    
  }
  while (( ch = socket->ReadChar()) != 0 )
  {
    if ( ch == '\n' )
    {
      break;
    }
  }

  nErrCode = TS_OK;
  while ( nErrCode == TS_OK )
  {
    TSDisplayField* pDisplayField = new TSDisplayField( *this );
    nErrCode = pDisplayField->Receive( socket );
    if ( nErrCode != TS_OK )
    {
      delete pDisplayField;
      break;
    }
    item.m_fieldList.push_back( pDisplayField );
  }
  while (( ch = socket->ReadChar()) != 0 )
  {
    if ( ch == '\n' )
    {
      break;
    }
  }

  nErrCode = TS_OK;
  while ( nErrCode == TS_OK )
  {
    TSItemLink* pItemLink = new TSItemLink( *this );
    nErrCode = pItemLink->Receive( socket );
    if ( nErrCode != TS_OK )
    {
      delete pItemLink;
      break;
    }
    item.m_linkList.push_back( pItemLink );
  }
  while (( ch = socket->ReadChar()) != 0 )
  {
    if ( ch == '\n' )
    {
      break;
    }
  }
  return TS_OK;
}

int TSServer::TSAuxiliaryItemCancelUpdate( TSAuxiliaryItem& auxItem )
{
  if ( !auxItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSAuxiliaryItemCancelUpdate&Params=";
  TSEncodeInt( auxItem.GetItemId(), params );
  s += params;
  TSEncodeInt( auxItem.GetTableId(), params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::TSAuxiliaryItemFinishUpdate( TSAuxiliaryItem& auxItem,
                                           bool bStealLockFlag /*=false*/ )
{
  if ( !auxItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSAuxiliaryItemFinishUpdate&Params=";
  TSEncodeInt( auxItem.GetItemId(), params );
  s += params;
  TSEncodeInt( auxItem.GetTableId(), params );
  s += params;
  TSEncodeInt( auxItem.m_nTransitionId, params );
  s += params;
  TSEncodeInt( (int)bStealLockFlag, params );
  s += params;

  TSDisplayFieldList dfList = auxItem.GetFieldList();
  TSDisplayFieldList::iterator dfIt = dfList.begin();
  int count = dfList.size();
  if ( !count )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  for ( ; dfIt != dfList.end(); dfIt++ )
  {
    if ( (*dfIt)->m_bModifiedFlag )
    {
      TSEncodeInt( (*dfIt)->GetItemId(), params );
      s += params;
      if ( (*dfIt)->m_bUseInternal )
      {
        TSEncodeString( (*dfIt)->m_sInternalValue, params );
        s += params;
      }
      else
      {
        TSEncodeString( (*dfIt)->GetDisplayValue(), params );
        s += params;
      }
    }
  }
  s += "\r\n";

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  return nResult;
}

int TSServer::TSAuxiliaryItemRead( TSAuxiliaryItem& auxItem )
{
  if ( !auxItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }
  
  TSString s;
  TSString params;
  s = "Func=TSAuxiliaryItemRead&Params=";
  TSEncodeInt( auxItem.GetItemId(), params );
  s += params;
  TSEncodeInt( auxItem.GetTableId(), params );
  s += params;
  TSEncodeInt( auxItem.GetSectionMask(), params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  auxItem.m_sItemName = ""; // set to empty before receiving at end of existing string
  TSDeleteStdList< TSAttachment    >( auxItem.GetAttachmentList() );
  TSDeleteStdList< TSChangeHistory >( auxItem.m_changeList );
  TSDeleteStdList< TSDisplayField  >( auxItem.GetFieldList() );
  TSDeleteStdList< TSItemLink      >( auxItem.GetLinkList() );

  if ( ReceiveTSItem( m_pSocket, auxItem ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::TSAuxiliaryItemStartUpdate( TSAuxiliaryItem& auxItem,
                                          bool             bLockFlag /*=true*/ )
{
  if ( !auxItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSAuxiliaryItemStartUpdate&Params=";
  TSEncodeInt( auxItem.GetItemId(), params );
  s += params;
  TSEncodeInt( auxItem.GetTableId(), params );
  s += params;
  TSEncodeInt( int( bLockFlag ), params );
  s += params;
  
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( nResult != TS_OK )
  {
    return nResult;
  }

  auxItem.m_sItemName = ""; // Set to empty before receiving. Receive appends to end of existing string.
  TSDeleteStdList< TSDisplayField >( auxItem.GetFieldList() );

  if ( ReceiveTSItem( m_pSocket, auxItem ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::TSItemAddAttachment( TSItem&         item,
                                   int             nType,
                                   const TSString& sLabel,
                                   const TSString& sText )
{
  if ( !item.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  int stringLen;
  TSString params;
  long fileLen = 0;
  char* pBuf = NULL;

  s = "Func=TSItemAddAttachment&Params=";
  TSEncodeInt( item.GetItemId(), params );
  s += params;
  TSEncodeInt( item.GetTableId(), params );
  s += params;
  TSPrimaryItem* tempItem = dynamic_cast<TSPrimaryItem*>( &item );
  if ( tempItem )
  {
    TSEncodeInt( tempItem->GetItemNumber(), params );
    s += params;
    TSEncodeInt( tempItem->GetProjectId(), params );
    s += params;
  }
  else
  {
    TSEncodeInt( 0, params );
    s += params;
    TSEncodeInt( 0, params );
    s += params;
  }
  TSEncodeInt( nType, params );
  s += params;
  TSEncodeString( sLabel, params );
  s += params;

  // Append the length and file data and send them.
  stringLen = s.Length();
  if ( nType & TS_ATTACHATTRIB_FILE )
  {
    // Read the file directly into the string buffer.
    FILE* fpin;

    if ( (fpin = fopen( sText, "rb") ) == NULL )
    {
      TSSetLastError( TS_ITEM_NOT_FOUND );
      return TSGetLastError();
    }

    // Determine the file length.
    if ( fseek( fpin, 0L, SEEK_END ) != 0 )
    {
      fclose( fpin );
      TSSetLastError( TS_ERROR );
      return TSGetLastError();
    }
      
    if ( (fileLen = ftell( fpin )) < 0 )
    {
      fclose( fpin );
      TSSetLastError( TS_ERROR );
      return TSGetLastError();
    }

    // Extract just the file name.
    TSString sSpec = sText;
    char* p = sSpec.GetBuffer();
    char* q;
    if ( ( (q = strrchr( p, '\\' )) != 0 ) ||
         ( (q = strrchr( p, '/' )) != 0 ) )
    {
      p = q + 1;
    }
    TSString sName( p );

    TSEncodeString( sName, params );
    s += params;

    // Encode the file length.  
    TSEncodeInt( fileLen, params );
    s += params;

    // Open the buffer in the TSString.
    stringLen = s.Length();
    pBuf = s.GetBuffer( stringLen + fileLen );
    pBuf += stringLen;

    // Seek back to the beginning.
    if ( fseek( fpin, 0L, SEEK_SET ) != 0 )
    {
      fclose( fpin );
      TSSetLastError( TS_ERROR );
      return TSGetLastError();
    }

    // Read the whole thing at once.
    if ( (long)fread( pBuf, sizeof( char ), fileLen, fpin ) != fileLen )
    {
      fclose( fpin );
      TSSetLastError( TS_ERROR );
      return TSGetLastError();
    }

    if ( ferror( fpin ) )
    {
      fclose( fpin );
      TSSetLastError( TS_ERROR );
      return TSGetLastError();
    }

    fclose( fpin );
    stringLen += fileLen;
  }
  else
  {
    TSEncodeString( sText, params );
    s += params;
    stringLen = s.Length();
  }

  if ( !Send( m_pSocket, s, stringLen ) )
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }
  
  // receive the new attachment
  TSAttachment* pAttachment = new TSAttachment( *this );
  if ( pAttachment->Receive( m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  // We're not going to send the file attachment to the server and then
  // ask for the same bytes to be sent back again. We'll just malloc the
  // member and copy the buffer we already have on this side.
  if ( nType & TS_ATTACHATTRIB_FILE )
  {
    pAttachment->m_pFileData = new char[ fileLen ];
    memcpy( pAttachment->m_pFileData, pBuf, fileLen );
    pAttachment->m_nFileLength = fileLen;
  }

  item.m_attachmentList.push_back( pAttachment );
  return TS_OK;
}

int TSServer::TSItemAddItemLink( TSItem&         item,
                                 int             nDestTableId,
                                 int             nDestItemId,
                                 int             nLinkType,
                                 int             nDestProjectId,
                                 int             nDestItemNumber )
{
  if ( !item.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSItemAddItemLink&Params=";
  TSEncodeInt( item.GetItemId(), params );
  s += params;
  TSEncodeInt( item.GetTableId(), params );
  s += params;
  TSPrimaryItem* tempItem = dynamic_cast<TSPrimaryItem*>( &item );
  if ( tempItem )
  {
    TSEncodeInt( tempItem->GetItemNumber(), params );
    s += params;
    TSEncodeInt( tempItem->GetProjectId(), params );
    s += params;
  }
  else
  {
    TSEncodeInt( 0, params );
    s += params;
    TSEncodeInt( 0, params );
    s += params;
  }
  TSEncodeInt( nDestItemId, params );
  s += params;
  TSEncodeInt( nDestTableId, params );
  s += params;
  TSEncodeInt( nDestItemNumber, params );
  s += params;
  TSEncodeInt( nDestProjectId, params );
  s += params;
  TSEncodeInt( nLinkType, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  // receive the new item link
  TSItemLink* pItemLink = new TSItemLink( *this, 0 );
  if ( pItemLink->Receive( m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  item.m_linkList.push_back( pItemLink );
  return TS_OK;
}

int TSServer::TSItemDeleteAttachment( int  nAttachmentId,
                                      bool bDeleteBothSides /*=true*/ )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSItemDeleteAttachment&Params=";
  TSEncodeInt( nAttachmentId, params );
  s += params;
  TSEncodeInt( int( bDeleteBothSides ), params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::TSItemFinishSubmit( TSItem&    item, 
                                  int&       nItemNumber )
{
  if ( !item.GetTableId() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSItemFinishSubmit&Params=";
  TSEncodeInt( item.GetItemId(), params );
  s += params;
  TSEncodeInt( item.GetTableId(), params );
  s += params;
  TSPrimaryItem* tempPrimaryItem = dynamic_cast<TSPrimaryItem*>( &item );
  if ( tempPrimaryItem )
  {
    if ( !tempPrimaryItem->GetProjectId() )
    {
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    TSEncodeInt( tempPrimaryItem->GetProjectId(), params );
    s += params;
  }
  else
  {
    TSEncodeInt( 0, params );
    s += params;
  }
  TSEncodeInt( item.m_nTransitionId, params );
  s += params;
    
  TSDisplayFieldList dfList = item.GetFieldList();
  TSDisplayFieldList::iterator dfIt = dfList.begin();
  int count = dfList.size();
  if ( !count )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  for ( ; dfIt != dfList.end(); dfIt++ )
  {
    if ( (*dfIt)->m_bModifiedFlag )
    {
      TSEncodeInt( (*dfIt)->GetItemId(), params );
      s += params;
      if ( (*dfIt)->m_bUseInternal )
      {
        TSEncodeString( (*dfIt)->m_sInternalValue, params );
        s += params;
      }
      else
      {
        TSEncodeString( (*dfIt)->GetDisplayValue(), params );
        s += params;
      }
    }
  }
  s += "\r\n";
  
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( nResult )
  {
    if ( m_pSocket->ReceiveInt( &nItemNumber ) != TS_OK )
    {
      //TSErrorCode is set in ReceiveInt()
      return TSGetLastError();
    }
  }

  return nResult;
}

int TSServer::TSItemStartSubmit( TSItem& item )
{
  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSItemStartSubmit&Params=";
  TSPrimaryItem* tempPrimaryItem = dynamic_cast<TSPrimaryItem*>( &item );
  if ( tempPrimaryItem )
  {
    if ( !tempPrimaryItem->GetProjectId() )
    {
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    TSEncodeInt( 0, params );
    s += params;
    TSEncodeInt( tempPrimaryItem->GetProjectId(), params );
    s += params;
  }
  else
  {
    if ( !item.GetTableId() )
    {
      TSSetLastError( TS_INVALID_PARAMETERS );
      return TSGetLastError();
    }
    TSEncodeInt( item.GetTableId(), params );
    s += params;
    TSEncodeInt( 0, params );
    s += params;
  }
    
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( nResult != TS_OK )
  {
    return nResult;
  }

  item.m_sItemName = ""; // set to empty before receiving at end of existing string
  TSDeleteStdList< TSDisplayField >( item.GetFieldList() );// clear field list before receiving

  if ( ReceiveTSItem( m_pSocket, item ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveTSItem()
    return TSGetLastError();
  }

  if ( tempPrimaryItem )
  {
    if ( tempPrimaryItem->Receive( m_pSocket ) != TS_OK )
    {
      // TSErrorCode is set in Receive()
      return TSGetLastError();
    }
  }
  
  return TS_OK;
}

int TSServer::TSPrimaryItemCancelTransition( TSPrimaryItem& primItem )
{
  if ( !primItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSPrimaryItemCancelTransition&Params=";
  TSEncodeInt( primItem.GetItemId(), params );
  s += params;
  TSEncodeInt( primItem.GetTableId(), params );
  s += params;
  TSEncodeInt( primItem.GetItemNumber(), params );
  s += params;
  TSEncodeInt( primItem.GetProjectId(), params );
  s += params;
  
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::TSPrimaryItemFinishTransition( TSPrimaryItem& primItem,
                                             bool bStealLockFlag /*=false*/ )
{
  if ( !primItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSPrimaryItemFinishTransition&Params=";
  TSEncodeInt( primItem.GetItemId(), params );
  s += params;
  TSEncodeInt( primItem.GetTableId(), params );
  s += params;
  TSEncodeInt( primItem.GetProjectId(), params );
  s += params;
  TSEncodeInt( primItem.m_nTransitionId, params );
  s += params;
  TSEncodeInt( (int)bStealLockFlag, params );
  s += params;

  TSDisplayFieldList dfList = primItem.GetFieldList();
  TSDisplayFieldList::iterator dfIt = dfList.begin();
  int count = dfList.size();
  if ( !count )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }
  for ( ; dfIt != dfList.end(); dfIt++ )
  {
    if ( (*dfIt)->m_bModifiedFlag )
    {
      TSEncodeInt( (*dfIt)->GetItemId(), params );
      s += params;
      if ( (*dfIt)->m_bUseInternal )
      {
        TSEncodeString( (*dfIt)->m_sInternalValue, params );
        s += params;
      }
      else
      {
        TSEncodeString( (*dfIt)->GetDisplayValue(), params );
        s += params;
      }
    }
  }
  s += "\r\n";
  
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  return nResult;
}

int TSServer::TSPrimaryItemGetTransitionList( TSPrimaryItem&            primItem,
                                              std::list<TSTransition*>& transList )
{
  if ( !primItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSPrimaryItemGetTransitionList&Params=";
  TSEncodeInt( primItem.GetItemId(), params );
  s += params;
  TSEncodeInt( primItem.GetTableId(), params );
  s += params;
  TSEncodeInt( primItem.GetItemNumber(), params );
  s += params;
  TSEncodeInt( primItem.GetProjectId(), params );
  s += params;
  
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  int nErrCode = TS_OK;
  while ( nErrCode == TS_OK )
  {
    TSTransition* pTransition = new TSTransition( *this );
    nErrCode = pTransition->Receive( m_pSocket );
    if ( nErrCode != TS_OK )
    {
      delete pTransition;
      break;
    }
    transList.push_back( pTransition );
  }
  
  return TS_OK;
}

int TSServer::TSPrimaryItemRead( TSPrimaryItem& primItem )
{
  if ( !primItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSPrimaryItemRead&Params=";
  TSEncodeInt( primItem.GetItemId(), params );
  s += params;
  TSEncodeInt( primItem.GetTableId(), params );
  s += params;
  TSEncodeInt( primItem.GetSectionMask(), params );
  s += params;
  TSEncodeInt( primItem.GetItemNumber(), params );
  s += params;
  TSEncodeInt( primItem.GetProjectId(), params );
  s += params;
  
  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  primItem.m_sItemName = ""; // set to empty before receiving at end of existing string
  TSDeleteStdList< TSAttachment    >( primItem.GetAttachmentList() );
  TSDeleteStdList< TSChangeHistory >( primItem.m_changeList );
  TSDeleteStdList< TSDisplayField  >( primItem.GetFieldList() );
  TSDeleteStdList< TSItemLink      >( primItem.GetLinkList() );

  if ( ReceiveTSItem( m_pSocket, primItem ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveTSItem()
    return TSGetLastError();
  }
  if ( primItem.Receive( m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::TSPrimaryItemStartTransition( TSPrimaryItem&  primItem,
                                            int             nTransitionId,
                                            const TSString& sTransitionName /*=TSString()*/,
                                            bool            bLockFlag /*=true*/ )
{
  if ( !primItem.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSPrimaryItemStartTransition&Params=";
  TSEncodeInt( primItem.GetItemId(), params );
  s += params;
  TSEncodeInt( primItem.GetTableId(), params );
  s += params;
  TSEncodeInt( primItem.GetItemNumber(), params );
  s += params;
  TSEncodeInt( primItem.GetProjectId(), params );
  s += params;
  TSEncodeInt( nTransitionId, params );
  s += params;
  TSEncodeString( sTransitionName, params );
  s += params;
  TSEncodeInt( (int)bLockFlag, params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( nResult != TS_OK )
  {
    return nResult;
  }

  primItem.m_sItemName = ""; // set to empty before receiving at end of existing string
  TSDeleteStdList< TSDisplayField >( primItem.GetFieldList() );

  if ( ReceiveTSItem( m_pSocket, primItem ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveTSItem()
    return TSGetLastError();
  }
  if ( primItem.Receive( m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::TSProjectRead( TSProject& proj )
{
  if ( !proj.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSProjectRead&Params=";
  TSEncodeInt( proj.GetItemId(), params );
  s += params;
  TSString fullName = proj.GetFullProjectName();
  TSEncodeString( fullName , params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  proj.m_sFullProjectName = ""; // set to empty before receiving at end of existing string
  proj.m_sItemName = "";        // set to empty before receiving at end of existing string
  if ( proj.Receive( m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

int TSServer::TSRecordRefRead( TSRecordRef& recRef )
{
  if ( !recRef.IsValid() )
  {
    TSSetLastError( TS_INVALID_PARAMETERS );
    return TSGetLastError();
  }

  if ( !GetSocket() )
  {
    // TSErrorCode is set in Open(Get)Socket
    return TSGetLastError();
  }

  TSString s;
  TSString params;
  s = "Func=TSRecordRefRead&Params=";
  TSEncodeInt( recRef.GetItemId(), params );
  s += params;
  TSEncodeInt( recRef.GetTableId(), params );
  s += params;

  if ( !Send( m_pSocket, s ))
  {
    // TSErrorCode is set in Send()
    return TSGetLastError();
  }

  int nResult;
  if ( m_pSocket->ReceiveInt( &nResult ) != TS_OK )
  {
    // TSErrorCode is set in ReceiveInt()
    return TSGetLastError();
  }

  if ( !nResult )
  {
    TSSetLastError( TS_ERROR );
    return TSGetLastError();
  }

  recRef.m_sItemName = "";        // set to empty before receiving at end of existing string
  if ( recRef.Receive( m_pSocket ) != TS_OK )
  {
    // TSErrorCode is set in Receive()
    return TSGetLastError();
  }

  return TS_OK;
}

