// TSServer.h: interface for the TSServer and TSSchema classes.
//
/////////////////////////////////////////////////////////////////////////

#ifndef TSSERVER_H_
#define TSSERVER_H_

#include "TSDef.h"
#include "TSList.h"
#include "TSSocket.h"
#include "TSString.h"

// Disable some annoying STL warnings.
#pragma warning( push, 1 )
#include "DisableSTLWarnings.h"
#include <list>
#include <vector>
#pragma warning( pop )

class TSAttachment;
class TSAuxiliaryItem;
class TSChangeHistory;
class TSDisplayField;
class TSItem;
class TSItemLink;
class TSObject;
class TSProject;
class TSPrimaryItem;
class TSRecord;
class TSRecordRef;
class TSTransition;

typedef std::list<TSProject*> TSProjectList;

class AFX_EXT_CLASS TSSchema : public TSObject
{
  public:
    virtual ~TSSchema();
    virtual TSObject* NewObject();
    virtual TSObject* Duplicate( int type = 0 );
    virtual void Copy( TSObject* obj );
    virtual TSString StringDump( int recursive, TSString indentation );
    virtual int SocketString( TSString& str );

    int tableId;
    int userDefinedFields;
    TSString name;
    TSFieldList fieldList;
};

class AFX_EXT_CLASS TSServer
{
  public:
    TSServer();
    ~TSServer();

    // Connect to the TeamTrack  server.
    int Connect( const char* userName,
                 const char* password,
                 const char* serverAddress,
                 const char* proxyAddress = NULL );

    // Accessors
    const char* GetDllWebAddress();
    const char* GetLastErrorMessage();

    //////////////////////////////////////////////////////////////////////////////////
    // Database access functions.  You *MUST* call Connect()
    // before you can call any of these functions.
    //////////////////////////////////////////////////////////////////////////////////

    // Retrieve the schema for a table. This is the schema pointer cached in the server.
    // Please don't modify it or delete it.  Needs Connect() in case not cached.
    TSSchema* GetSchema( int tableId );
    
    int AddField( TSRecord* field, int tableId, int fieldType, int& fieldId );
    int AddRecord( TSRecord* rec, TSRecord* newRec = NULL );
    int AddTable( TSRecord* table, TSRecord* solution = NULL, int optionalFields = 0, int reserved = 0 );
    int BuildFieldList( int tableId, TSFieldList& fieldList ); // Allocates a field list of all of the fields for this table.
    int ClearAlternateUser();
    int CreateIndex( int tableId, TSList *columnNames, const char* indexName );
    int DeleteIndex( int tableId, const char* indexName );
    int DeleteRecord( int tableId, int id );
    int ExecuteSQL( TSString& sql );
    int GetConnectionInfo( char* dsn, char* databaseName, char* serverName );
    int GetConnectionInfo( char** dsn, char** databaseName, char** serverName );
    int GetDbInfo( int infotype, void* out );
    int GetInt( int tblid, const char* columnName, int recId, int* pInt );
    int GetPrivilegeName( enum privId_e privId, char** privName );
    int GetString( int tblid, const char* columnName, int recId, char* string, int size );
    int GetString( int tblid, const char* columnName, int recId, char** string );
    int GetSubmitTransition( int projectId, int* id );
    int GetTableIdByDatabaseNameEx( const char* tableDbName, int* tableId );
    int HasPrivilege( int userId, int projectId, int maskNumber, int mask ); // Deprecated. Use HasUserPrivilege or HasGroupPrivilege.
    int HasGroupPrivilege( int groupId, int itemId, enum privId_e privId );  // Use this HasPrivilege method.
    int HasUserPrivilege( int userId, int itemId, enum privId_e privId );    // Use this HasPrivilege method.
    int HasRecordPrivilege( enum recPriv_e recPriv, int userId, int tableId, int recId );
    int HasValidLicense( const char* solutionIdent );
    int MoveFolder( int folderId, int newParentId, int position );
    int MoveProject( int projectId, int newParentId, int position );
    int ReadAllRecords( TSRecordList* list, int tableId );
    int ReadAttachmentList( TSRecordList* list, int tableId, int recId );
    int ReadAvailableTransitionList( TSRecordList* list, int tableId, int recId );
    int ReadChangeList( TSRecordList* list, int caseId, BOOL newFirst = FALSE );
    int ReadFieldsByProject( TSRecordList* list, int projectId );
    int ReadFolderItems( TSRecordList* list, int folderId );
    int ReadFolderList( TSRecordList* list, int owner, int parentId );
    int ReadProjectSelectionList( TSRecordList* list, int selectId );
    int ReadProjectTransitionList( TSRecordList* list, int transitionId );
    int ReadPropertyList( TSRecordList* list, int transId );
    int ReadRecord( TSRecord* record, int id );
    int ReadRecordByItemNumber( TSRecord* pRec, TSString sItemNumber, int nProjectId );
    int ReadRecordForId( TSRecord* record, const char* searchId );
   	int ReadRecordListWithWhere( TSRecordList* recordList, int tableId, const char* whereClause );
   	int ReadRecordListWithWhere( TSRecordList* recordList,           // Reads a record list,
                                 int tableId,                        // filtering by the where
                                 const char* whereClause,            // clause, returning only
                                 const std::vector< int >& fields ); // the desired fields.
  	int ReadRecordWithWhere( TSRecord* record, const char* whereClause );
    int ReadRecordWithWhere( TSRecord* record,                  // Reads a record, filtering
                             const char* whereClause,           // by the where clause, returning 
                             const std::vector< int >& fields );// only the desired fields.
    int ReadReport( TSRecord* report, const char* name );
    int ReadReportList( TSRecordList* list, long userId, int perm );
    int ReadSelectionList( TSRecordList* list, int fieldId, int projectId );
    int ReadStateList( TSRecordList* list, int projectId, BOOL incParent = FALSE );
    int ReadTransitionList( TSRecordList* list, int projectId );
    int ReadUserDefinedFields( TSRecordList* list, int tableId );
    int ReadUserListForPrivilege( TSRecordList* list,
                                  int userid,
                                  TSRecordList* fieldList,
                                  int mask );
    int ReadUserSelectionList( TSRecordList* list, // This is deprecated.  Do not use.
                               int fieldId,
                               int projectId,
                               BOOL incDeleted = FALSE );
    int ReadVCActions( TSRecordList* list, int caseId );
    int ReadVCActionsForModule( TSRecordList* list,
                                const char* filename,
                                int userid,
                                int action2 = 0 );
    int RefreshCache( const int tableId );
    int ReleaseRecordLockById( int nTableId, int nItemId );
    int ReleaseRecordLockByNumber( int nProjectId, int nItemNumber );
    int SetAlternateUser( const char* loginId,
                          const char* pwd,
                          bool bValidatePwd = true );
    int SetExitUrl( const char* exitUrl );
    int SetGroupPrivilege( int groupId, int itemId, privId_e privid, bool bGrant = true );
    int SetUserPrivilege( int userId, int itemId, privId_e privid, bool bGrant = true );
    int SetRecordLockById( int nTableId, int nItemId,
                           bool bStealLockFlag = false );
    int SetRecordLockByNumber( int nProjectId, int nItemNumber,
                               bool bStealLockFlag = false );
    int Submit( int* nIssueId, TSString& sLoginid, TSRecord* pRec, int nTableId, // This is deprecated. Use
                int nProjectId, int nFolderId, int nType );                      // the other overloads.
    int Submit( int* nIssueId, TSString& sLoginid, TSRecord* pRec, int nTableId, // Use this Submit.
                int nProjectId, int nFolderId = 0 );
    int Submit( int *nId, int* nIssueId, TSString& sLoginid, TSRecord* pRec,     // Use this Submit to
                int nTableId, int nProjectId, int nFolderId = 0 );               // return the ts_id.
    int Transition( TSString& sLoginid,                                          
                    TSRecord* pRec,
                    int nProjectId,
                    int nTableId,
                    int nRecordId,
                    int nTransition );
    int UpdateList( TSRecordList* list, int tableId, int recid1 = 0, int recid2 = 0 );
    int UpdateRecord( TSRecord* rec, TSRecord* newRec = NULL );
    int ValidateUser( const char* loginId,
                      const char* pwd,
                      BOOL validatePwd = TRUE,
                      int* userId = NULL );
    int ValidateVersion();

    //////////////////////////////////////////////////////////////////////////////////
    // Methods pertaining to Display Data
    //////////////////////////////////////////////////////////////////////////////////
    int  GetProjectList( int nTableId, int nProjectsMask, TSProjectList& projectList );
    int  ReceiveTSItem( TSSocket* socket, TSItem& item );
    int  TSAuxiliaryItemCancelUpdate( TSAuxiliaryItem& auxItem );
    int  TSAuxiliaryItemFinishUpdate( TSAuxiliaryItem& auxItem, bool bStealLockFlag = false );
    int  TSAuxiliaryItemRead( TSAuxiliaryItem& auxItem );
    int  TSAuxiliaryItemStartUpdate( TSAuxiliaryItem& auxItem, bool bLockFlag = true );
    int  TSItemAddAttachment( TSItem& item, int nType, const TSString& sLabel, const TSString& sText );
    int  TSItemAddItemLink( TSItem& item, int nDestTableId, int nDestItemId, int nLinkType,
                            int DestProjectId, int DestItemNumber );
    int  TSItemDeleteAttachment( int nAttachmentId, bool bDeleteBothSides = true );
    int  TSItemFinishSubmit( TSItem& item, int& nItemNumber );
    int  TSItemStartSubmit( TSItem& item );
    int  TSPrimaryItemCancelTransition( TSPrimaryItem& primItem );
    int  TSPrimaryItemFinishTransition( TSPrimaryItem& primItem, bool bStealLockFlag = false );
    int  TSPrimaryItemGetTransitionList( TSPrimaryItem& primItem, std::list<TSTransition*>& transList );
    int  TSPrimaryItemRead( TSPrimaryItem& primItem );
    int  TSPrimaryItemStartTransition( TSPrimaryItem& primItem,
                                       int nTransitionId,
                                       const TSString& sTransitionName = TSString(),
                                       bool bLockFlag = true );
    int  TSProjectRead( TSProject& proj );
    int  TSRecordRefRead( TSRecordRef& recRef );

  private:
    // Member variables set by ::Connect used for communications
    TSString protocolString;  // protocol used to connect with (http or https)
    TSString serverName;      // name of the server connected to
    TSString directoryName;   // usually tmtrack
    TSString dllName;         // usually tmtrack.dll?
    int      portNumber;      // port number to connect to (80 or 443)

    TSString proxyString;     // proxy to be used to connect to the web server
    
    TSString authString;      // Basic Authentication string with encoded user:password
    TSString dllWebAddress;   // serverName/directoryName/dllName

    // Other member variables
    TSString errorMsg;
    TSList   schemaCache;
    TSString alternateUser;

    TSString EncodePassword( const char* password );
    TSSocket* OpenSocket();
    TSSocket* GetSocket();
    int Send( TSSocket* socket, const char* s, int len = 0 );
    TSSocket* m_pSocket;

    int ReleaseRecordLock( int nTableId, int nItemId,
                           int nProjectId = 0, int nItemNumber = 0 );
    int SetRecordLock( int nTableId, int nItemId,
                       bool bStealLockFlag,
                       int nProjectId = 0, int nItemNumber = 0 );

};

void TSSetLastError( int errorCode );
void TSClearLastError();
int  TSGetLastError();

#endif
