// TSString.cpp: implementation of the TSString class.
//
/////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>
#include <ctype.h>
#include <malloc.h>
#include <string.h>
#include "TSDef.h"
#include "TSString.h"

int TSString::BUFFER_EXCESS = 10;

static char TS_EMPTY_STRING[] = "";

TSString::TSString( const char* str )
        : buffer( NULL ),
          bufsize( 0 ),
          length( 0 )
{
  operator = ( str );
}

TSString::TSString( const TSString& str )
        : buffer( NULL ),
          bufsize( 0 ),
          length( 0 )
{
  operator = ( str.buffer );
}

TSString::TSString()
        : buffer( NULL ),
          bufsize( 0 ),
          length( 0 )
{
}

TSString::~TSString()
{
  if ( buffer )
  {
    free( buffer );
  }
  buffer = NULL;
}

void TSString::Empty()
{
  if ( !buffer )
  {
    return;
  }
  
  free( buffer );
  buffer = NULL;
  length = 0;
}

void TSString::TrimRight( const char* str /*=NULL*/ )
{
  if ( !buffer || Length() == 0 )
  {
    return;
  }
  
  char* ptr = buffer + ( Length() - 1 ); // go to the end of the buffer
  
  if ( str ) // trim the specified characters from the buffer
  {
    while ( *ptr )
    {
      char* strPtr = ( char* )str;
      while ( *strPtr && *ptr != *strPtr )
      {
        strPtr++;
      }
      if ( *strPtr == '\0' )
      {
        break; // no match; quit the outer loop
      }
      ptr--;
    }
  }
  else
  {
    // move left of any whitespace characters
    while ( iswspace( *ptr ))
    {
      ptr--;
    }
  }

  *( ++ptr ) = '\0';
  length = strlen( buffer );
}

void TSString::TrimRight( const char c )
{
  if ( !buffer || Length() == 0 )
  { 
    return;
  }

  char* ptr = buffer + (Length() - 1);

  while ( ptr >= buffer  && *ptr == c )
  {
    ptr--;
  }

  *( ++ptr ) = '\0';
  length = strlen( buffer );
}

void TSString::TrimRight( const TSString& str )
{
  TrimRight( str.buffer );
}

void TSString::TrimLeft( const char* str /*=NULL*/ )
{
  if ( !buffer || Length() == 0 )
  {
    return;
  }

  char* ptr = buffer; // start from the beginning of the buffer
  
  if ( str ) // trim the specified characters from the buffer
  {
    while ( *ptr )
    {
      char* strPtr = ( char* )str;
      while ( *strPtr && *ptr != *strPtr )
      {
        strPtr++;
      }
      if ( *strPtr == '\0' )
      {
        break; // no match; quit the outer loop
      }
      ptr++;
    }
  }
  else // trim white space
  {
     while ( iswspace( *ptr ))
    {
      ptr++;
    }
  }

  if ( buffer != ptr )
  {
    memmove( buffer, ptr, strlen( ptr ) + 1 );
  }

  length = strlen( buffer );
}

void TSString::TrimLeft( const char c )
{
  if ( !buffer || Length() == 0 )
  {
    return;
  }

  char* ptr = buffer;

  while ( *ptr == c )
  {
    ptr++;
  }

  if ( buffer != ptr )
  {
    memmove( buffer, ptr, strlen( ptr ) + 1 );
  }

  length = strlen( buffer );
}

void TSString::TrimLeft( const TSString& str )
{
  TrimLeft( str.buffer );
}

TSString& TSString::operator += ( const char* str )
{
  if ( str != NULL )
  {
    if ( buffer )
    {
      int len = strlen( str );
      if ( length + 1 + len > bufsize )
      {
        bufsize = length + 1 + len + TSString::BUFFER_EXCESS;
        buffer = static_cast<char*>( realloc( buffer, sizeof( char )*bufsize ));
      }
      strcat( buffer, str );
      length += len;
    }
    else
    {
      operator = ( str );
    }
  }
  return *this;
}

TSString& TSString::operator += ( const TSString& str )
{
  operator += ( str.buffer );
  return *this;
}

TSString& TSString::operator += ( char c )
{
  if ( buffer )
  {
    if ( length + 1 < bufsize )
    {
      buffer[length] = c;
      buffer[length+1] = '\0';
    }
    else
    {
      bufsize = length + 2 + TSString::BUFFER_EXCESS;
      buffer = static_cast<char*>( realloc( buffer, sizeof( char )*bufsize ));
      buffer[length] = c;
      buffer[length+1] = '\0';
    }
  }
  else
  {
    bufsize = 2 + TSString::BUFFER_EXCESS;
    buffer = static_cast<char*>( malloc( sizeof( char)*bufsize ));
    buffer[0] = c;
    buffer[1] = '\0';
  }
  length++;
  return *this;
}

TSString& TSString::operator = ( const char* str )
{
  if ( buffer )
  {
    free( buffer );
    buffer = NULL;
  }
  if ( str )
  {
    length = strlen( str );
    bufsize = length + TSString::BUFFER_EXCESS + 1;
    buffer = static_cast<char*>( malloc( sizeof( char )*bufsize ));
    strcpy( buffer, str );
  }
  return *this;
}

TSString& TSString::operator = ( const TSString& str )
{
  operator = ( str.buffer );
  return *this;
}

BOOL TSString::operator == ( const TSString& str ) const
{
  return operator==( str.buffer );
}

BOOL TSString::operator == ( const char* str ) const
{
  if ( buffer == NULL && str == NULL )
  {
    return TRUE;
  }

  if ( buffer == NULL )
  {
    if ( str[ 0 ] == '\0' )
    { // Null and empty string should still match
      return TRUE;
    }
    
    return FALSE;
  }

  if ( str == NULL )
  {
    if ( buffer[ 0 ] == '\0' )
    { // Null and empty string should still match
      return TRUE;
    }
    
    return FALSE;
  }

  return !strcmp( buffer, str );
}

BOOL TSString::operator != ( const TSString& str ) const
{
  return !(*this == str.buffer);
}

BOOL TSString::operator != ( const char* str ) const
{
  return !(*this == str);
}

char TSString::operator [] ( int idx ) const
{
  if ( !buffer )
  { 
    return 0;
  }
  if ( idx > Length() )
  {
    return 0;
  }
  else
  {
    return buffer[idx];
  }
}

TSString TSString::operator + ( const TSString& second )
{
  TSString tmp( *this );
  tmp += second;
  return tmp;
}

TSString TSString::operator + ( const char* str )
{
  TSString tmp( *this );
  tmp += str;
  return tmp;
}

TSString::operator const char*() const
{
  if ( buffer )
  {
    return buffer;
  }
  else
  {
    return TS_EMPTY_STRING;
  }
}

TSObject* TSString::NewObject()
{
  return new TSString;
}

void TSString::Copy( TSObject* sourceString )
{
  operator = ((static_cast<TSString*>( sourceString ))->buffer );
}

TSObject* TSString::Duplicate( int /*type = 0*/ )
{
  return new TSString( buffer );
}

TSString TSString::StringDump( int /*recursive*/, TSString indentation )
{
  return ( indentation + *this );
}

int TSString::SocketString( TSString& str )
{
  str = *this;
  return TS_OK;
}

int TSString::CompareNoCase( const char* str ) const
{
  if ( buffer == NULL && str == NULL )
  {
    return 0;
  }

  if ( buffer == NULL )
  {
    if ( str[ 0 ] == '\0' )
    { // Null and empty string should still match
      return 0;
    }
    
    return -1;
  }

  if ( str == NULL )
  {
    if ( buffer[ 0 ] == '\0' )
    { // Null and empty string should still match
      return 0;
    }
    
    return 1;
  }

#ifdef _WIN32
  return stricmp( buffer, str );
#else
  return strcasecmp( buffer, str );
#endif
}

char* TSString::Find( const char* str ) const
{
  if ( buffer == NULL || str == NULL )
  {
    return NULL;
  }

  return strstr( buffer, str );
}

char* TSString::GetBuffer()
{
  return GetBuffer( 1 );
}

char* TSString::GetBuffer( size_t minsize )
{
  if ( buffer )
  {
    if ( bufsize < minsize + TSString::BUFFER_EXCESS )
    {
      bufsize = minsize + TSString::BUFFER_EXCESS;
      buffer = static_cast<char*>( realloc( buffer, sizeof( char )*bufsize ));
    }
  }
  else
  {
    bufsize = minsize + TSString::BUFFER_EXCESS;
    buffer = static_cast<char*>( malloc( sizeof( char )*bufsize ));
    buffer[ 0 ] = '\0';
  }

  return buffer;
}
