#             Perforce Defect Tracking Integration Project
#              <http://www.ravenbrook.com/project/p4dti/>
#
#  SCHEMA-REMARKS.PY -- REMARKS FOR BUGZILLA SCHEMA DOCUMENTATION
#
#             Nick Barnes, Ravenbrook Limited, 2003-07-07
#
#
# 1. INTRODUCTION
#
# This module contains data structures holding remarks concerning
# the Bugzilla schema.  These remarks are automatically included in the
# Bugzilla schema doc by the code in  make_schema_doc.py.
#
# The intended readership is project developers.
#
# This document is not confidential.

# this function constructs HTML referring to a column in a table.

def the_column(table, column):
    return '<a href="#column-%s-%s">%s.%s</a>' % (table, column, table, column)

# this function constructs HTML for a remark identifying a column as
# referring to a column of another table.

def foreign_key(table, column):
    return '(foreign key ' + the_column(table, column) + ')'

# this function constructs HTML for a remark identifying a column as
# equivalent to a field of another table.

def the_table(table):
    return 'the <a href="#table-%s">"%s"</a> table' % (table, table)

# This is a map from table name to an HTML remark concerning that
# table, which is output before the schema for that table.
#
# Tables with no attached remarks are given 'None' as a placeholder, so
# we know to add a remark later.

table_remarks = {
    'attachments': None,

    'attachstatusdefs': None,

    'attachstatuses': None,

    'bug_group_map': None,

    'bugs': None,

    'bugs_activity': None,

    'cc': None,

    'components': 'One row for each component.  See <a href="#notes-products">the notes on products and components.</a>',

    'dependencies': None,

    'duplicates': 'This table identifies bugs which are duplicates of other bugs.',

    'fielddefs': 'This is a table of the fields for which we update ' + the_table('bugs_activity') + '.  It\'s used to turn bugs_activity entries into useful text.',

    'flagexclusions': None,

    'flaginclusions': None,

    'flags': None,

    'flagtypes': None,

    'group_control_map': None,

    'group_group_map': None,

    'groups': 'This table describes a number of user groups.  Each group allows its members to perform a restricted activity.  See <a href="#notes-groups">the notes on groups</a>. ',

    'keyworddefs': 'Names and definitions of the keywords.  See <a href="#notes-keywords">the notes on keywords</a>.',

    'keywords': 'Bugs may have keywords.  This table defines which bugs have which keywords.  The keywords are defined in ' + the_table('keyworddefs') + '.',

    'logincookies': 'Bugzilla generates a cookie each time a user logs in, and uses it for subsequent authentication.  The cookies generated are stored in this table.  For more information, see <a href="#notes-authentication">the notes on authentication</a>.',

    'longdescs': None,

    'milestones': None,

    'namedqueries': None,

    'products': 'One row for each product.  See <a href="#notes-products">the notes on products.</a>',

    'profiles': 'This table describes Bugzilla users.  One row per user.',

    'profiles_activity': 'This table is for recording changes to ' + the_table('profiles') + '. Currently it only records changes to ' + the_column('profiles', 'groupset') + ' made with editusers.cgi.  This allows the administrator to track group inflation.  There is currently no code to inspect this table; only to add to it.',

    'quips': None,

    'series': None,

    'series_categories': None,
    
    'series_data': None,

    'shadowlog': None,

    'tokens': 'Used for managing the tokens sent to users who wish to change their password (or email address, <b>from 2.16</b>).',

    'user_group_map': None,

    'user_series_map': None,

    'versions': None,

    'votes': None,

    'watch': None,
}

table_added_remarks = {
    'attachstatusdefs': None,

    'attachstatuses': None,

    'bug_group_map': None,

    'duplicates': None,

    'flagexclusions': None,

    'flaginclusions': None,

    'flags': None,

    'flagtypes': None,

    'group_control_map': None,

    'group_group_map': None,

    'quips': None,

    'series': None,

    'series_categories': None,
    
    'series_data': None,

    'tokens': None,

    'user_group_map': None,

    'user_series_map': None,

}

table_removed_remarks = {
    'attachstatusdefs': 'replaced by the flag tables',

    'attachstatuses': ' replaced by the flag tables',

    'shadowlog': 'similar functionality now available using MySQL\'s replication facilities',

}

# This is a map from table name to a map from column name to HTML
# remark for that column.  At present, these remarks include schema
# change comments (which will eventually be generated automatically).
#
# Columns with no attached remarks are given 'None' as a placeholder,
# so we know to add a remark later.

column_remarks = {
    'attachments': {
        'attach_id': 'a unique ID.',

        'bug_id': 'the bug to which this is attached ' + foreign_key('bugs', 'bug_id'),

        'creation_ts': 'the creation time.',

        'description': 'a description of the attachment.',

        'mimetype': 'the MIME type of the attachment.',

        'ispatch': 'non-zero if this attachment is a patch file.',

        'isprivate': 'Non-zero if this attachment is "private", i.e. only visible to members of the "insider" group.',

        'isobsolete': 'Non-zero if this attachment is marked as obsolete.',

        'filename': 'the filename of the attachment.',

        'thedata': 'the content of the attachment.',

        'submitter_id': 'the userid of the attachment ' + foreign_key('profiles', 'userid'),

        },

    'attachstatusdefs': {
        'id': 'a unique ID.',

        'name': 'the name of the attachment status.',

        'description': 'The description of the attachment status.',

        'sortkey': 'A number used to determine the order in which attachment statuses are shown.',

        'product': 'The product for which bugs can have attachments with this status ' + foreign_key('products', 'product'),

        },

    'attachstatuses': {

        'attach_id': 'The id of the attachment ' + foreign_key('attachments', 'attach_id'),

        'statusid': 'The id of the status ' + foreign_key('attachstatusdefs', 'id'),

        },

    'bug_group_map': {

        'bug_id': None,

        'group_id': None,

        },

    'bugs': {

        'bug_id': 'The bug ID.',

        'groupset': 'The groups which this bug occupies. Each group corresponds to one bit. See the <a href="#table-groups">"groups" table</a>.',

        'assigned_to': 'The current owner of the bug.',

        'bug_file_loc': 'A URL which points to more information about the bug.',

        'bug_severity': 'See the <a href="#notes-severity">notes</a>.',

        'bug_status': 'The <a href="#notes-workflow">workflow</a> status of the bug.',

        'creation_ts': 'The times of the bug\'s creation.',

        'delta_ts': 'The timestamp of the last update.  This includes updates to some related tables (e.g. ' + the_table('longdescs') + ').',

        'short_desc': 'A short description of the bug.',

        'op_sys': 'The operating system on which the bug was observed.',

        'priority': 'The priority of the bug (P1 = most urgent, P5 = least urgent).',

        'product': 'The product ' + foreign_key('products', 'product'),

        'product_id': 'The product ' + foreign_key('products', 'id'),

        'rep_platform': 'The platform on which the bug was reported.',

        'reporter': 'The user who reported this ' + foreign_key('profiles', 'userid'),

        'version': 'The product version ' + foreign_key('versions', 'value'),

        'component': 'The product component ' + foreign_key('components', 'value'),

        'component_id': 'The product component ' + foreign_key('components', 'id'),

        'resolution': 'The bug\'s <a href="#notes-workflow">resolution</a>',

        'target_milestone': 'The milestone by which this bug should be resolved.  ' + foreign_key('milestones', 'value'),

        'qa_contact': 'The QA contact ' + foreign_key('profiles', 'userid'),

        'status_whiteboard': 'This seems to be just a small whiteboard field.',

        'votes': 'The number of votes.',

        'keywords': 'A set of keywords.  Note that this duplicates the information in ' + the_table('keywords') + '. ' + foreign_key('keyworddefs', 'name'),

        'lastdiffed': 'The time at which information about this bug changing was last emailed to the cc list.',
        
        'everconfirmed': '1 if this bug has ever been confirmed.  This is used for validation of some sort.',

        'reporter_accessible': '1 if the reporter can see this bug (even if in the wrong group); 0 otherwise.',

        'assignee_accessible': '1 if the assignee can see this bug (even if in the wrong group); 0 otherwise.',
        
        'qacontact_accessible': '1 if the QA contact can see this bug (even if in the wrong group); 0 otherwise.',

        'cclist_accessible': '1 if people on the CC list can see this bug (even if in the wrong group); 0 otherwise.',

        'estimated_time': None,
        
        'remaining_time': None,
        
        'alias': None,
        

        },

    'bugs_activity': {

        'bug_id': 'Which bug ' + foreign_key('bugs', 'bug_id'),

        'who': 'Which user ' + foreign_key('profiles', 'userid'),

        'bug_when': 'When was the change made?',

        'fieldid': 'What was the fieldid? ' + foreign_key('fielddefs', 'fieldid'),

        'attach_id': 'If the change was to an attachment, the ID of the attachment ' + foreign_key('attachments', 'attach_id'),

        'oldvalue': 'The head of the old value.',

        'newvalue': 'The head of the new value.',

        'added': 'The new value of this field, or values which have been added for multi-value fields such as ' + the_column('bugs', 'keywords') + ', ' + the_table('cc') + ', and ' + the_table('dependencies'),

        'removed': 'The old value of this field, or values which have been removed for multi-value fields such as ' + the_column('bugs', 'keywords') + ', ' + the_table('cc') + ', and ' + the_table('dependencies'),

        },

    'cc': {

        'bug_id': 'The bug ' + foreign_key('bugs', 'bug_id'),

        'who': 'The user ' + foreign_key('profiles', 'userid'),

        },

    'components': {

        'name': 'The component id.',

        'id': 'The component id.',

        'value': 'The component name.',

        'program': 'The product ' + foreign_key('products', 'product'),

        'product_id': 'The product ' + foreign_key('products', 'id'),

        'initialowner': 'The default initial owner of bugs in this component.  On component creation, this is set to the user who creates the component. <b>Before 2.12: foreign key ' + the_column('profiles', 'login_name') + '; from 2.12: foreign key ' + the_column('profiles', 'userid') + '.</b>',

        'initialqacontact': 'The initial "qa_contact" field for bugs of this component. Note that the use of the qa_contact field is optional, parameterized by Param("useqacontact").  <b>Before 2.12: foreign key ' + the_column('profiles', 'login_name') + '; from 2.12: foreign key ' + the_column('profiles', 'userid') + '.</b>',

        'description': 'A description of the component.',

        },
    
    'dependencies': {

        'blocked': 'Which bug is blocked ' + foreign_key('bugs', 'bug_id'),

        'dependson': 'Which bug does it depend on ' + foreign_key('bugs', 'bug_id'),

        },

    'duplicates': {
    
        'dupe_of': 'The bug which is duplicated ' + foreign_key('bugs', 'bug_id'),
    
        'dupe': 'The duplicate bug ' + foreign_key('bugs', 'bug_id'),

        },

    'fielddefs': {
    
        'fieldid': 'primary key for this table',

        'name': 'field name or definition (some fields are names of other tables or of fields in other tables).',

        'description': 'long description',

        'mailhead': 'whether or not to send the field description in mail notifications.',

        'sortkey': 'the order of fields in mail notifications.',

        },

    'flagexclusions': {

    'type_id': None,

    'product_id': None,

    'component_id': None,
    },

    'flaginclusions': {

    'type_id': None,

    'product_id': None,

    'component_id': None,
    },

    'flags': {

    'id': None,

    'type_id': None,

    'status': None,

    'bug_id': None,

    'attach_id': None,

    'creation_date': None,

    'modification_date': None,

    'setter_id': None,

    'requestee_id': None,
    },

    'flagtypes': {

    'id': None,

    'name': None,

    'description': None,

    'cc_list': None,

    'target_type': None,

    'is_active': None,

    'is_requestable': None,

    'is_requesteeble': None,

    'is_multiplicable': None,

    'sortkey': None,
    },

    'group_control_map': {

    'group_id': None,

    'product_id': None,

    'entry': None,

    'membercontrol': None,

    'othercontrol': None,

    'canedit': None,
    },

    'group_group_map': {

    'member_id': None,

    'grantor_id': None,

    'isbless': None,
    },

    'groups': {

    'bit': '2^n for some n.  Assigned automatically.',

    'name': 'A short name for the group.',

    'description': 'A long description of the group.',

    'isbuggroup': '1 if this is a group controlling access to a set of bugs.',

    'userregexp': 'a regexp used to determine membership of new users.',

    'isactive': '1 if bugs can be added to this group; 0 otherwise.',

    'id': None,

    'last_changed': None,

    },

    'keyworddefs': {

    'id': 'A unique number identifying this keyword.',

    'name': 'The keyword itself.',

    'description': 'The meaning of the keyword.',
    },

    'keywords': {

    'bug_id': 'The bug ' + foreign_key('bugs', 'bug_id'),

    'keywordid': 'The keyword ID ' + foreign_key('keyworddefs', 'id'),
    },

    'logincookies': {

    'cookie': 'The cookie',

    'userid': 'The user id; ' + foreign_key('profiles', 'userid'),

    'cryptpassword': 'The encrypted password used on this login.',

    'hostname': 'The CGI REMOTE_HOST for this login.',

    'ipaddr': 'The CGI REMOTE_ADDR for this login.',

    'lastused': 'The timestamp of this login.',
    },

    'longdescs': {

    'bug_id': 'the bug ' + foreign_key('bugs', 'bug_id'),

    'who': 'the user who added this text ' + foreign_key('profiles', 'userid'),

    'bug_when': 'when the text was added',

    'thetext': 'the text itself.',

    'isprivate': 'Non-zero if this comment is "private", i.e. only visible to members of the "insider" group.',

    'work_time': None
        
    },

    'milestones': {

    'value': 'The name of the milestone (e.g. "3.1 RTM", "0.1.37", "tweakfor BigCustomer", etc).',

    'product': 'The product ' + foreign_key('products', 'product'),

    'product_id': 'The product ' + foreign_key('products', 'id'),

    'sortkey': 'A number used for sorting milestones for a given product.',
    },

    'namedqueries': {

    'userid': 'The user whose query this is.',

    'name': 'The name of the query.',

    'watchfordiffs': 'Unused.',

    'linkinfooter': 'Whether or not the query should appear in the foot of every page.',

    'query': 'The query (text to append to the query page URL).',
    },

    'products': {

    'product': 'The name of the product.',

    'id': 'The product ID.',

    'name': 'The product name.',

    'description': 'The description of the product',

    'milestoneurl': 'The URL of a document describing the product milestones.',

    'disallownew': 'New bugs can only be created for this product if this is 0.',

    'votesperuser': 'Total votes which a single user has for bugs of this product.',

    'maxvotesperbug': 'Maximum number of votes which a bug may have.',

    'votestoconfirm': 'How many votes are required for this bug to become NEW.',

    'defaultmilestone': 'The default milestone for a new bug.',
    },

    'profiles': {

    'userid': 'A unique identifier for the user.  Used in other tables to identify this user.',

    'login_name': 'The user\'s email address.  Used when logging in or providing mailto: links.',

    'password': 'The user\'s password, in plaintext.',

    'cryptpassword': 'The user\'s password.  Before 2.14, the MySQL function <code>encrypt</code> was used to encrypt passwords.  From 2.14, the Perl function <code>crypt</code> is used instead.',

    'realname': 'The user\'s real name.',

    'groupset': 'The set of groups to which the user belongs.  Each group corresponds to one bit and confers powers upon the user. See the <a href="#table-groups">"groups" table</a>.',

    'emailnotification': 'Controls when email reporting bug changes is sent to this user.',

    'disabledtext': 'If non-empty, indicates that this account has been disabled and gives a reason. ',

    'newemailtech': 'is non-zero if the user wants to user the "new" email notification technique.',

    'mybugslink': 'indicates whether a "My Bugs" link should appear at the bottom of each page.',

    'blessgroupset': 'Indicates the groups into which this user is able to introduce other users.',

    'emailflags': 'Flags controlling when email messages are sent to this user.',

    'refreshed_when': None,

    },

    'profiles_activity': {

    'userid': 'The profile which has changed ' + foreign_key('profiles', 'userid'),

    'who': 'The user who changed it ' + foreign_key('profiles', 'userid'),

    'profiles_when': 'When it was changed',

    'fieldid': 'The ID of the changed field ' + foreign_key('fielddefs', 'fieldid'),

    'oldvalue': 'The old value',

    'newvalue': 'The new value.',
    },

    'quips': {

    'quipid': None,

    'userid': None,

    'quip': None,

    'approved': None,
    },

    'series': {

    'series_id': None,

    'creator': None,

    'category': None,

    'subcategory': None,

    'name': None,

    'frequency': None,

    'last_viewed': None,

    'query': None,
    },

    'series_categories': {

    'category_id': None,

    'name': None,
    },

    'series_data': {

    'series_id': None,

    'date': None,

    'value': None
    },

    'shadowlog': {

    'id': 'unique id',

    'ts': 'timestamp',

    'reflected': '0',

    'command': 'SQL command',
    },

    'tokens': {

    'userid': 'The user to whom the token was issued.  ' + foreign_key('profiles', 'userid'),

    'issuedate': 'The date at which the token was issued',

    'token': 'The token itself.',

    'tokentype': 'The type of the token.  <b>Before 2.16:</b> only "password".<b>From 2.16:</b> also "emailold" and "emailnew".',

    'eventdata': None,
    },

    'user_group_map': {

    'user_id': None,

    'group_id': None,

    'isbless': None,

    'isderived': None,
    },

    'user_series_map': {

    'user_id': None,

    'series_id': None,
    },

    'versions': {

    'value': 'The name of the version',

    'program': 'The product ' + foreign_key('products', 'product'),

    'product_id': 'The product ' + foreign_key('products', 'id'),
    },

    'votes': {

    'who': 'The user ' + foreign_key('profiles', 'userid'),

    'bug_id': 'The bug ' + foreign_key('bugs', 'bug_id'),

    'count': 'How many votes.',

    'vote_count': 'How many votes.',
    },

    'watch': {

    'watcher': 'The watching user ' + foreign_key('profiles', 'userid'),

    'watched': 'The watched user ' + foreign_key('profiles', 'userid'),
    },
}

# This is a map from table name to a map from column name to HTML
# remark for that column.  At present, these remarks include schema
# change comments (which will eventually be generated automatically).
#
# Columns with no attached remarks are given 'None' as a placeholder,
# so we know to add a remark later.

column_added_remarks = {
    'attachments': {
        'isprivate': None,

        'isobsolete': None,

        },

    'bugs': {

        'alias': None,

        'product_id': 'replacing "product"',

        'component_id': 'replacing "component"',

        'reporter_accessible': None,

        'assignee_accessible': None,
        
        'qacontact_accessible': None,

        'cclist_accessible': None,

        'estimated_time': None,

        'remaining_time': None,

        },

    'bugs_activity': {

        'attach_id': None,

        'added': 'replacing "newvalue"',

        'removed': 'replacing "oldvalue"',

        },

    'components': {

        'name': 'replacing "value"',

        'id': 'replacing "value" as the primary key',

        'product_id': 'replacing "program"',

        },

    'groups': {

    'id': None,

    'isactive': None,

    'last_changed': None,

    },

    'logincookies': {

    'ipaddr': 'replacing hostname',

    },

    'longdescs': {

    'isprivate': None,

    'work_time': None,

    },

    'milestones': {

    'product_id': 'replacing "product"',

    },

    'products': {

    'id': 'replacing "product" as the table key',

    'name': 'replacing "product" as the product name',

    },

    'profiles': {

    'emailflags': None,

    'refreshed_when': None,

    },

    'quips': {

    'approved': None,

    },

    'versions': {

    'product_id': 'replacing "program"',
    },

    'votes': {

    'vote_count': 'replacing "count"',
    },

}

# This is a map from table name to a map from column name to HTML
# remark for that column.  At present, these remarks include schema
# change comments (which will eventually be generated automatically).
#
# Columns with no attached remarks are given 'None' as a placeholder,
# so we know to add a remark later.

column_removed_remarks = {
    'bugs': {

        'groupset': None,

        'product': 'replaced by "product_id"',

        'component': 'replaced by "component_id"',

        'assignee_accessible': None,
        
        'qacontact_accessible': None,

        },

    'bugs_activity': {

        'oldvalue': 'replaced by "removed"',

        'newvalue': 'replaced by "added"',

        },

    'components': {

        'value': 'replaced by "name" and "id"',

        'program': 'replaced by "product_id"',

        },
    
    'groups': {

        'bit': 'replaced by "id"',

        },
    
    'logincookies': {

    'cryptpassword': None,

    'hostname': 'replaced by "ipaddr"',

    },

    'milestones': {

    'product': 'replaced by "product_id"',

    },

    'namedqueries': {

    'watchfordiffs': None,

    },

    'products': {

    'product': 'replaced with "id" and "name"',

    },

    'profiles': {

    'password': None,

    'groupset': None,

    'emailnotification': 'replaced in part by "emailflags"',

    'newemailtech': None,

    'blessgroupset': None,

    },

    'versions': {

    'program': 'replaced by "product_id"',

    },

    'votes': {

    'count': 'replaced by "vote_count"',
    },

}

# This is a map from table name to a map from index name to HTML
# remark for that index.  At present, these remarks include schema
# change comments (which will eventually be generated automatically).
#
# Indexes with no attached remarks are given 'None' as a placeholder,
# so we know to add a remark later.

index_remarks = {
    'attachments': {
        'PRIMARY': None,
        'bug_id': None,
        'creation_ts': None,
        },
    'attachstatusdefs': {
        'PRIMARY': None,
        },
    'attachstatuses': {
        'PRIMARY': None,
        },
    'bug_group_map': {
        'bug_id': None,
        'group_id': None,
        },
    'bugs': {
        'PRIMARY': None,
        'alias': None,
        'assigned_to': None,
        'creation_ts': None,
        'delta_ts': None,
        'bug_severity': None,
        'bug_status': None,
        'op_sys': None,
        'priority': None,
        'product': None,
        'product_id': None,
        'reporter': None,
        'short_desc': None,
        'version': None,
        'component': None,
        'component_id': None,
        'resolution': None,
        'target_milestone': None,
        'qa_contact': None,
        'votes': None,
        },
    'bugs_activity': {
        'bug_id': None,
        'bug_when': None,
        'fieldid': None,
        },
    'cc': {
        'who': None,
        'bug_id': None,
        },
    'components': {
        'PRIMARY': None,
        'product_id': None,
        'name': None,
        'bug_id': None,
        'bug_when': None,
        'fieldid': None,
        },
    'dependencies': {
        'blocked': None,
        'dependson': None,
        },
    'duplicates': {
        'PRIMARY': None,
        },
    'fielddefs': {
        'PRIMARY': None,
        'name': None,
        'sortkey': None,
        },
    'flagexclusions': {
        'type_id': None,
        },
    'flaginclusions': {
        'type_id': None,
        },
    'flags': {
        'PRIMARY': None,
        'bug_id': None,
        'setter_id': None,
        'requestee_id': None,
        },
    'flagtypes': {
        'PRIMARY': None,
        },
    'group_control_map': {
        'product_id': None,
        'group_id': None,
        },
    'group_group_map': {
        'member_id': None,
        },
    'groups': {
        'PRIMARY': None,
        'bit': None,
        'name': None,
        },
    'keyworddefs': {
        'PRIMARY': None,
        'name': None,
        },
    'keywords': {
        'keywordid': None,
        'bug_id': None,
        },
    'logincookies': {
        'PRIMARY': None,
        'lastused': None,
        },
    'longdescs': {
        'bug_id': None,
        'who': None,
        'bug_when': None,
        'thetext': None,
        },
    'milestones': {
        'product': None,
        'product_id': None,
        },
    'namedqueries': {
        'userid': None,
        'watchfordiffs': None,
        },
    'products': {
        'PRIMARY': None,
        'name': None,
        },
    'profiles': {
        'PRIMARY': None,
        'login_name': None,
        },
    'profiles_activity': {
        'userid': None,
        'profiles_when': None,
        'fieldid': None,
        },
    'quips': {
        'PRIMARY': None,
        },
    'series': {
        'PRIMARY': None,
        'creator': None,
        'creator_2': None,
        },
    'series_categories': {
        'PRIMARY': None,
        'name': None,
        },
    'series_data': {
        'series_id': None,
        },
    'shadowlog': {
        'PRIMARY': None,
        'reflected': None,
        },
    'tokens': {
        'PRIMARY': None,
        'userid': None,
        },
    'user_group_map': {
        'user_id': None,
        },
    'user_series_map': {
        'user_id': None,
        'series_id': None,
        },
    'versions': {
        },
    'votes': {
        'who': None,
        'bug_id': None,
        },
    'watch': {
        'watched': None,
        'watcher': None,
        },
    }

index_removed_remarks = {
    'bugs': {
        'product': 'replaced by "product_id"',
        'component': 'replaced by "component_id"',
        },

    'groups': {
        'bit': 'replaced by "PRIMARY"',
        },

    'milestones': {
        'product': 'replaced by "product_id"',
        },

    'namedqueries': {
        'watchfordiffs': None,
        },

    }

index_added_remarks = {
    'bugs': {
        'alias': None,
        'component_id': 'replacing "component"',
        'product_id': 'replacing "product"',
        'short_desc': None,
    },

    'components': {
        'PRIMARY': None,
        'name': None,
        'product_id': None,
        },

    'groups': {
        'PRIMARY': 'replacing "bit"',
        },
    
    'longdescs': {
        'who': None,
        'thetext': None,
        },

    'milestones': {
        'product_id': 'replacing "product"',
        },

    'products': {
        'PRIMARY': None,
        'name': None,
        },
    
    }

# This prelude is included in the generated schema doc prior to the
# schema itself.

prelude="""<?xml version="1.0" encoding="UTF-8"?>

<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "DTD/xhtml1-transitional.dtd">

<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">

<head>

<title>Bugzilla database schema</title>

</head>

<body bgcolor="#FFFFFF" text="#000000" link="#000099" vlink="#660066" alink="#FF0000">

<div align="center">

<p>
<a href="/">Ravenbrook</a> /
<a href="/project/">Projects</a> /
<a href="/project/p4dti/">Perforce Defect Tracking Integration</a> /
<a href="/project/p4dti/version/2.1/">Version 2.1 Product Sources</a> /
<a href="/project/p4dti/version/2.1/design/">Design</a>
</p>

<p><i><a href="/project/p4dti/">Perforce Defect Tracking Integration Project</a></i></p>

<hr />


<h1>Bugzilla database schema</h1>

<address>
<a href="mailto:nb@ravenbrook.com">Nick Barnes</a>,
<a href="http://www.ravenbrook.com/">Ravenbrook Limited</a>,
2000-11-14
</address>

</div>


<h2><a id="section-1" name="section-1">1. Introduction</a></h2>

<p>This document describes the Bugzilla database schema for Bugzilla
versions 2.10, 2.12, 2.14, 2.14.1, 2.14.2, 2.14.3, 2.14.4, 2.14.5,
2.16rc1, 2.16rc2, 2.16, 2.16.1, 2.16.2, 2.16.3, 2.16.4, 2.17.1,
2.17.3, 2.17.4, 2.17.5, and 2.17.6.</p>

<p>This document is now generated automatically by a Python script
which constructs and colours the schema tables from the results of
MySQL queries.</p>

<p>The purpose of this document is to act as a reference while
developing P4DTI code which interacts with Bugzilla.</p>

<p>The intended readership is P4DTI developers.</p>

<p>This document is not confidential.</p>

<h2><a id="section-2" name="section-2">2. Bugzilla overview</a></h2>

<p>Bugzilla is a defect tracking system, written in Perl with a CGI
web GUI.  It uses MySQL to store its tables.  We need to understand
Bugzilla and MySQL to build a P4DTI Bugzilla integration.</p>

<h3><a id="notes-releases" name="notes-releases">Bugzilla Releases</a></h3>

<p>This table gives the dates of all the Bugzilla releases since 2.0,
and of each P4DTI release which changed the set of supported Bugzilla
releases, and shows the history of P4DTI support for each release.</p>

<table border="1" cellspacing="0" cellpadding="5">

<thead>

  <tr align="left">

    <th>Date</th>

    <th>Release</th>

    <th>First P4DTI Release</th>

    <th>Last P4DTI Release</th>

    <th>Notes</th>
  </tr>

</thead>

<tbody>

  <tr valign="top" align="left">

    <td>1998-09-19</td>

    <td>2.0</td>

    <td>-</td>

    <td>-</td>

    <td>Never supported by the P4DTI.  Not described in this document.</td>

  </tr>

  <tr valign="top" align="left">

    <td>1999-02-10</td>

    <td>2.2</td>

    <td>-</td>

    <td>-</td>

    <td>Never supported by the P4DTI.  Not described in this document.</td>

  </tr>

  <tr valign="top" align="left">

    <td>1999-04-30</td>

    <td>2.4</td>

    <td>-</td>

    <td>-</td>

    <td>Never supported by the P4DTI.  Not described in this document.</td>

  </tr>

  <tr valign="top" align="left">

    <td>1999-08-30</td>

    <td>2.6</td>

    <td>-</td>

    <td>-</td>

    <td>Never supported by the P4DTI.  Not described in this document.</td>

  </tr>

  <tr valign="top" align="left">

    <td>1999-11-19</td>

    <td>2.8</td>

    <td>-</td>

    <td>-</td>

    <td>Never supported by the P4DTI.  Not described in this document.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2000-05-09</td>

    <td>2.10</td>

    <td>1.0.0</td>

    <td>1.5.3</td>

    <td>The first Bugzilla release supported by the P4DTI.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2001-02-19</td>

    <td colspan="4">P4DTI release 1.0.0, supporting Bugzilla 2.10.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2001-04-27</td>

    <td>2.12</td>

    <td>1.1.2</td>

    <td>1.5.3</td>

    <td>&nbsp;</td>

  </tr>

  <tr valign="top" align="left">

    <td>2001-07-16</td>

    <td colspan="4">P4DTI release 1.1.2, supporting Bugzilla 2.10 and 2.12.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2001-08-29</td>

    <td>2.14</td>

    <td>1.2.0</td>

    <td>1.5.3</td>

    <td>&nbsp;</td>

  </tr>

  <tr valign="top" align="left">

    <td>2001-10-18</td>

    <td colspan="4">P4DTI release 1.2.0, supporting Bugzilla 2.10, 2.12, and 2.14.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-01-05</td>

    <td>2.14.1</td>

    <td>1.4.0</td>

    <td>1.5.3</td>

    <td>A security patch release.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-02-04</td>

    <td colspan="4">P4DTI release 1.4.0, supporting Bugzilla 2.10, 2.12, 2.14, and 2.14.1.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-05-10</td>

    <td>2.16rc1</td>

    <td>-</td>

    <td>-</td>

    <td>A release candidate.  Never supported by the P4DTI.  Schema is as for 2.16.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-06-07</td>

    <td>2.16rc2</td>

    <td>-</td>

    <td>-</td>

    <td>A release candidate.  Never supported by the P4DTI.  Schema is as for 2.16.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-06-07</td>

    <td>2.14.2</td>

    <td>-</td>

    <td>-</td>

    <td>A security patch release.  Never formally supported by the P4DTI.  Use 2.14.5 or 2.16.3 instead.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-07-28</td>

    <td>2.16</td>

    <td>-</td>

    <td>-</td>

    <td>Never formally supported by the P4DTI. Use 2.16.3 instead.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-07-28</td>

    <td>2.14.3</td>

    <td>-</td>

    <td>-</td>

    <td>A security patch release.  Never formally supported by the P4DTI.  Use 2.14.5 or 2.16.3 instead.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-09-30</td>

    <td>2.16.1</td>

    <td>1.5.0</td>

    <td>-</td>

    <td>A security patch release.  Deprecated in P4DTI 2.0 (use 2.16.3 instead).</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-09-30</td>

    <td>2.14.4</td>

    <td>1.5.0</td>

    <td>-</td>

    <td>A security patch release.  The first Bugzilla-on-Windows release
  supported by the P4DTI.  Deprecated in P4DTI 2.0 (use 2.14.5 or
  2.16.3 instead).</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-11-01</td>

    <td colspan="4">P4DTI release 1.5.0, supporting Bugzilla 2.14.4 and 2.16.1.  Bugzilla 2.14.4 supported on Windows.  Bugzilla 2.10, 2.12, 2.14, 2.14.1 deprecated.  Patches also supplied for Bugzilla 2.14.2, 2.14.3, and 2.16.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2002-11-25</td>

    <td>2.17.1</td>

    <td>-</td>

    <td>-</td>

    <td>A development release.  Never supported by the P4DTI.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-01-02</td>

    <td>2.16.2</td>

    <td>2.0.0</td>

    <td>-</td>

    <td>A security patch release.  Deprecated in P4DTI 2.0 (use 2.16.3 instead).</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-01-02</td>

    <td>2.14.5</td>

    <td>2.0.0</td>

    <td>-</td>

    <td>A security patch release.  Currently supported by the P4DTI on Windows, Linux, and Unix.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-01-02</td>

    <td>2.17.3</td>

    <td>-</td>

    <td>-</td>

    <td>A development release.  Never supported by the P4DTI.  There was no release 2.17.2.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-04-25</td>

    <td>2.16.3</td>

    <td>2.0.0</td>

    <td>-</td>

    <td>Currently supported by the P4DTI on Linux and Unix only.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-04-25</td>

    <td>2.17.4</td>

    <td>-</td>

    <td>-</td>

    <td>A development release.  Never supported by the P4DTI.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-06-02</td>

    <td colspan="4">P4DTI release 2.0.0, supporting Bugzilla 2.14.5 and 2.16.3.  Bugzilla 2.14.5 supported on Windows.  Bugzilla 2.14.4, 2.16.1, and 2.16.2 deprecated.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-11-03</td>

    <td>2.17.5</td>

    <td>-</td>

    <td>-</td>

    <td>A development release.  Never supported by the P4DTI.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-11-03</td>

    <td>2.16.4</td>

    <td>-</td>

    <td>-</td>

    <td>Not yet supported by the P4DTI.</td>

  </tr>

  <tr valign="top" align="left">

    <td>2003-11-10</td>

    <td>2.17.6</td>

    <td>-</td>

    <td>-</td>

    <td>A development release.  Never supported by the P4DTI.</td>

  </tr>

</tbody>
</table>


<p>The P4DTI currently supports Bugzilla 2.14.4, 2.14.5, 2.16.1,
2.16.2, 2.16.3, and 2.16.4.  During the lifetime of this document it
has also had some form of support for Bugzilla 2.10, 2.12, 2.14,
2.14.1, 2.14.2, 2.14.3, and 2.16.  It is intended to support 2.18 when
it arrives as the culmination of the 2.17.x series of releases.</p>

<h3><a id="notes-schema-changes" name="notes-schema-changes">Bugzilla Schema Changes</a></h3>

<p>Where the Bugzilla schema has been changed since Bugzilla 2.10, the
change is noted in this document.  We need to be able to recognize the
schema version.</p>

<p>In the schema tables themselves, changed fields are noted and
colored as follows:</p>

<table border="1" cellspacing="0" cellpadding="5">

  <tr bgcolor="#ffffff" valign="top" align="left">

    <td>A field whose definition and use which has not changed between
    2.10 and 2.17.6.</td>

  </tr>

  <tr bgcolor="#ffcccc" valign="top" align="left">

    <td>A field which was present in some previous Bugzilla release
    but which is absent from 2.17.6.</td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td>A field which is present in 2.17.6 but was absent in some
    previous Bugzilla release.</td>

  </tr>

  <tr bgcolor="#ccccff" valign="top" align="left">

    <td>A field whose definition has changed over time.</td>

  </tr>

</table>

<p>In Bugzilla release 2.12, the following schema changes were
made:</p>

<ul>
  <li>The <a href="#table-duplicates">duplicates</a> table was
  added.</li>

  <li><a href="#table-profiles">profiles</a>.emailflags was
  added.</li>

  <li>The <a href="#table-bugs">bugs</a>.resolution value <b>MOVED</b>
  was added.</li>

  <li>A number of additional values were permitted in <a
  href="#column-bugs-op_sys">bugs.op_sys</a>.</li>

  <li><a
  href="#column-components-initialowner">components.initialowner</a>
  and <a
  href="#column-components-initialqacontact">components.initialqacontact</a>
  changed from "tinytext" (foreign key <a
  href="#column-profiles-login_name">profiles.login_name</a>) to
  "mediumint" (foreign key <a
  href="#column-profiles-userid">profiles.userid</a>), default 0.</li>

  <li><a
  href="#column-profiles-disabledtext">profiles.disabledtext</a>
  changed from "not null" to "null".</li>

  <li>The default value of <a
  href="#column-profiles-newemailtech">profiles.newemailtech</a>
  changed from 0 to 1.</li>

</ul>

<p>In Bugzilla release 2.14, the following schema changes were
made:</p>

<ul>
  <li>The <a href="#table-tokens">tokens</a> table was
  added.</li>

  <li><a href="#column-profiles-password">profiles.password</a> was
  removed.</li>

  <li><a
  href="#column-profiles-cryptpassword">profiles.cryptpassword</a> and
  <a
  href="#column-logincookies-cryptpassword">logincookies.cryptpassword</a>
  were both changed from varchar(64) to varchar(32).</li>

  <li><a
  href="#column-profiles-newemailtech">profiles.newemailtech</a> was
  removed.</li>

  <li><a
  href="#column-profiles-emailnotification">profiles.emailnotification</a>
  was removed.</li>

  <li><a
  href="#column-bugs-reporter_accessible">bugs.reporter_accessible</a>,
  <a
  href="#column-bugs-assignee_accessible">bugs.assignee_accessible</a>,
  <a
  href="#column-bugs-qacontact_accessible">bugs.qacontact_accessible</a>,
  and <a
  href="#column-bugs-cclist_accessible">bugs.cclist_accessible</a>
  were added.</li>

  <li><a href="#column-bugs-version">bugs.version</a> changed from
  varchar(16) to varchar(64).</li>

  <li><a
  href="#column-bugs_activity-oldvalue">bugs_activity.oldvalue</a> and
  <a href="#column-bugs_activity-newvalue">bugs_activity.newvalue</a>
  were replaced by <a
  href="#column-bugs_activity-removed">bugs_activity.removed</a> and
  <a href="#column-bugs_activity-added">bugs_activity.added</a>.</li>

  <li><a href="#column-groups-isactive">groups.isactive</a> was
  added.</li>

  <li><a href="#column-longdescs-who">longdescs.who</a> became an
  index field.</li>

  <li><a
  href="#column-profiles-disabledtext">profiles.disabledtext</a>
  changed back to "not null".</li>

</ul>

<p>The schema is identical in Bugzilla releases 2.14 and 2.14.1.</p>

<p>In Bugzilla release 2.14.2, the following schema change was
made:</p>

<ul>

  <li><a
  href="#column-logincookies-hostname">logincookies.hostname</a> was
  replaced by <a
  href="#column-logincookies-ipaddr">logincookies.ipaddr</a>.</li>

</ul>

<p>The schema is identical in Bugzilla releases 2.14.2, 2.14.3,
2.14.4, and 2.14.5.</p>

<p>In Bugzilla release 2.16 (and the release candidates 2.16rc1 and
2.16rc2), the following schema changes were made:</p>

<ul>

  <li>The <a href="#table-attachstatuses">attachstatuses</a> and<a
  href="#table-attachstatusdefs">attachstatusdefs</a> tables were
  added.</li>

  <li><a
  href="#column-attachments-isobsolete">attachments.isobsolete</a> was
  added.</li>

  <li>The values permitted in <a
  href="#column-bugs-op_sys">bugs.op_sys</a> changed.</li>

  <li><a
  href="#column-bugs-assignee_accessible">bugs.assignee_accessible</a>
  and <a
  href="#column-bugs-qacontact_accessible">bugs.qacontact_accessible</a>
  were removed.</li>

  <li><a
  href="#column-bugs_activity-attach_id">bugs_activity.attach_id</a>
  was added.</li>

  <li><a
  href="#column-logincookies-cryptpassword">logincookies.cryptpassword</a>
  was removed.</li>

  <li>The possible values of <a
  href="#column-tokens-tokentype">tokens.tokentype</a> changed, to
  include 'emailold' and 'emailnew' (used when changing the email
  address of a Bugzilla user).</li>

</ul>

<p>The schema is identical in Bugzilla releases 2.16rc1, 2.16rc2,
2.16, 2.16.1, 2.16.2, 2.16.3, and 2.16.4.</p>

<p>In Bugzilla release 2.17.1, the following schema changes were
made:</p>

<p>## Note: I am still working on updating the schema remarks to
reflect all these changes.  Changes which have not yet been reflected
in the remarks are marked here with "##".</p>

<ul>

  <li><p>## The groups system was radically changed.  This included the
  following detailed schema changes:</p>
    <ul>

      <li>## The <a href="#table-bug_group_map">bug_group_map</a>, <a
      href="#table-user_group_map">user_group_map</a>, <a
      href="#table-group_group_map">group_group_map</a> were added.</li>

      <li>## <a href="#column-groups-bit">groups.bit</a> was replaced
      with <a href="#column-groups-id">groups.id</a>.</li>

      <li>## <a
      href="#column-groups-last_changed">groups.last_changed</a> was
      added.</li>

      <li><a href="#column-bugs-groupset">bugs.groupset</a>, <a
      href="#column-profiles-groupset">profiles.groupset</a> and <a
      href="#column-profiles-blessgroupset">profiles.blessgroupset</a>
      were dropped.</li> </ul> </li>

  <li>## The <a href="#table-attachstatuses">attachstatuses</a> and<a
  href="#table-attachstatusdefs">attachstatusdefs</a> tables were
  replaced with the <a href="#table-flags">flags</a>, <a
  href="#table-flagtypes">flagtypes</a>, <a
  href="#table-flaginclusions">flaginclusions</a>, <a
  href="#table-flagexclusions">flagexclusions</a> tables.  This
  replaces the "attachment statuses" feature introduced in 2.16 with a
  new "status flags" feature, allowing status flags to be defined and
  used on both attachments and bugs.</li>

  <li>The <a href="#table-quips">quips</a> table was added.</li

> <li>Products got IDs in addition to names, and product name columns
  were replaced with product ID columns: <a
  href="#column-bugs-product">bugs.product</a> was replaced with <a
  href="#column-bugs-product_id">bugs.product_id</a>, <a
  href="#column-components-program">components.program</a> was
  replaced with <a
  href="#column-components-product_id">components.product_id</a>, <a
  href="#column-milestones-product">milestones.product</a> was
  replaced with <a
  href="#column-milestones-product_id">milestones.product_id</a>, <a
  href="#column-versions-program">versions.program</a> was replaced
  with <a href="#column-versions-product_id">versions.product_id</a>,
  and <a href="#column-products-product">products.product</a> was
  replaced with <a href="#column-products-id">products.id</a> and <a
  href="#column-products-name">products.name</a>.</li>

  <li>Components got IDs in addition to names, and the component name
  column was replaced with a component ID column: <a
  href="#column-bugs-component">bugs.component</a> was replaced with
  <a href="#column-bugs-component_id">bugs.component_id</a>, and <a
  href="#column-components-value">components.value</a> was replaced
  with <a href="#column-components-id">components.id</a> and <a
  href="#column-components-name">components.name</a>.</li>

  <li>## <a
  href="#column-bugs-estimated_time">bugs.estimated_time</a>, <a
  href="#column-bugs-remaining_time">bugs.remaining_time</a>, and <a
  href="#column-longdescs-work_time">longdescs.work_time</a> were
  added.</li>

  <li><a
  href="#column-attachments-isprivate">attachments.isprivate</a> and
  <a href="#column-longdescs-isprivate">longdescs.isprivate</a> were
  added.</li>

  <li><a
  href="#column-attachments-creation_ts">attachments.creation_ts</a>
  changed from a timestamp to a datetime default '0000-00-00
  00:00:00'.</li>

  <li><a href="#column-attachments-filename">attachments.filename</a>
  changed from a mediumtext to a varchar(100).</li>

  <li>## The values permitted in <a
  href="#column-bugs-op_sys">bugs.op_sys</a> changed.</li>

  <li>## <a href="#column-bugs-alias">bugs.alias</a> was added.</li>

  <li>## <a
  href="#column-namedqueries-watchfordiffs">namedqueries.watchfordiffs</a>
  was added.</li>

  <li>## <a
  href="#column-profiles-refreshed_when">profiles.refreshed_when</a>
  was added.</li>

</ul>

<p>## We also need more extensive notes on some aspects of Bugzilla
which have been changed in this release, in particular the groups
system. ##</p>

<p>The schema is identical in Bugzilla releases 2.17.1 and 2.17.2.</p>

<p>In Bugzilla release 2.17.3, the following schema changes were
made:</p>

<p>## Note: I am still working on updating the schema remarks to
reflect all these changes.  Changes which have not yet been reflected
in the schema remarks are marked here with "##".</p>

<ul>

  <li>## The <a href="#table-group_control_map">group_control_map</a>
  table was added.</li>

  <li>The <a href="#table-shadowlog">shadowlog</a> table was
  removed.</li>

</ul>

<p>In Bugzilla release 2.17.4, the following schema changes were
made:</p>

<ul>

  <li><a href="#column-quips-approved">quips.approved</a> was
  added.</li>

</ul>

<p>In Bugzilla release 2.17.5, the following schema changes were
made:</p>

<p>## Note: I am still working on updating the schema remarks to
reflect all these changes.  Changes which have not yet been reflected
in the schema remarks are marked here with "##".</p>

<ul>

  <li>## The <a href="#table-series">series</a>, <a
  href="#table-series_categories">series_categories</a>, <a
  href="#table-series_data">series_data</a>, and <a
  href="#table-user_series_map">user_series_map</a> tables were
  added.</li>

  <li>## <a href="#column-votes-count">votes.count</a> was replaced
  with <a href="#column-votes-vote_count">votes.vote_count</a>.</li>

  <li>## The values permitted in <a
  href="#column-bugs-op_sys">bugs.op_sys</a> changed.</li>

  <li>## <a href="#column-bugs-short_desc">bugs.short_desc</a> became
  a fulltext index field.</li>

  <li>## <a href="#column-longdescs-thetext">longdescs.thetext</a> became
  a fulltext index field.</li>

</ul>

<p>The schema is identical in Bugzilla releases 2.17.5 and 2.17.6.</p>

<h3><a id="notes-bugs" name="notes-bugs">Bugs</a></h3>

<p>Each defect is called a <b>bug</b> and corresponds to one row in
the <a href="#table-bugs">bugs</a> table.  It is identified by its
number, <a href="#column-bugs-bug_id">bugs.bug_id</a>.</p>

<h3><a id="notes-products" name="notes-products">Products and components</a></h3>

<p>The work managed by Bugzilla is divided into products.  Each
product is represented by a row in the <a
href="#table-products">products</a> table. The work for each product
is in turn divided into the components of that product.  Each
component is represented by a row in the <a
href="#table-components">components</a> table.</p>

<p>Several properties of a new bug (e.g. ownership) are determined by
the product and component to which it belongs.</p>

<h3><a id="notes-workflow" name="notes-workflow">Workflow</a></h3>

<p>Each bug has a status (<a
href="#column-bugs-bug_status">bugs.bug_status</a>).  If a bug has a
status which shows it has been resolved, it also has a resolution (<a
href="#column-bugs-resolution">bugs.resolution</a>), otherwise the
resolution field is empty.  These tables show the possible values of
these fields and the valid transitions of the status field.</p>

<table border="1" cellspacing="0" cellpadding="5">
  <tr valign="top" align="left">
  
    <th>Status</th>
  
    <th>Resolved?</th>
  
    <th>Description</th>
  
    <th>Transitions</th>

  </tr>

  <tr valign="top" align="left">

    <td>UNCONFIRMED</td>

    <td>No</td>

    <td>A new bug, when a product has voting</td>

    <td>to NEW by voting or confirmation<br />
        to ASSIGNED by acceptance<br />
        to RESOLVED by resolution<br />
    </td>

  </tr>

  <tr valign="top" align="left">

    <td>NEW</td>

    <td>No</td>

    <td>Recently added or confirmed</td>

    <td>to ASSIGNED by acceptance<br />
        to RESOLVED by analysis and maybe fixing<br />
        to NEW by reassignment<br />
    </td>

  </tr>

  <tr valign="top" align="left">

    <td>ASSIGNED</td>

    <td>No</td>

    <td>Has been assigned</td>

    <td>to NEW by reassignment<br />
        to RESOLVED by analysis and maybe fixing<br />
    </td>

  </tr>

  <tr valign="top" align="left">

    <td>REOPENED</td>

    <td>No</td>

    <td>Was once resolved but has been reopened</td>

    <td>to NEW by reassignment<br />
        to ASSIGNED by acceptance<br />
        to RESOLVED by analysis and maybe fixing<br />
    </td>

  </tr>

  <tr valign="top" align="left">

    <td>RESOLVED</td>

    <td>Yes</td>

    <td>Has been resolved (e.g. fixed, deemed unfixable, etc.  See "resolution" column)</td>

    <td>to REOPENED by reopening<br />
        to VERIFIED by verification<br />
        to CLOSED by closing<br />
    </td>

  </tr>

  <tr valign="top" align="left">

    <td>VERIFIED</td>

    <td>Yes</td>

    <td>The resolution has been approved by QA</td>

    <td>to CLOSED when the product ships<br />
        to REOPENED by reopening<br />
    </td>

  </tr>

  <tr valign="top" align="left">

    <td>CLOSED</td>

    <td>Yes</td>

    <td>Over and done with</td>

    <td>to REOPENED by reopening</td>

  </tr>
</table>

<table border="1" cellspacing="0" cellpadding="5">
  <tr valign="top" align="left">
  
    <th>Resolution</th>
  
    <th>Meaning</th>

  </tr>

  <tr>

    <td>FIXED</td>

    <td>The bug has been fixed.</td>

  </tr>

  <tr>

    <td>INVALID</td>

    <td>The problem described is not a bug.</td>

  </tr>

  <tr>

    <td>WONTFIX</td>

    <td>This bug will never be fixed.</td>

  </tr>

  <tr>

    <td>LATER</td>

    <td>This bug will not be fixed in this version.</td>

  </tr>

  <tr>

    <td>REMIND</td>

    <td>This bug probably won't be fixed in this version.</td>

  </tr>

  <tr>

    <td>DUPLICATE</td>

    <td>This is a duplicate of an existing bug. (a description comment
        is added to this effect, and from 2.12 a record is added to
        the <a href="#table-duplicates">duplicates</a> table).</td>

  </tr>

  <tr>

    <td>WORKSFORME</td>

    <td>This bug could not be reproduced.</td>

  </tr>

  <tr bgcolor="#ccffcc">

    <td>MOVED</td>

    <td>This bug has been moved to another database. <b>Added in
    2.12.</b></td>

  </tr>
</table>

<h3><a id="notes-users" name="notes-users">Users</a></h3>

<p>Bugzilla has users.  Each user is represented by one row in the <a
href="#table-profiles">profiles</a> table.  Each user is referred by a
number (<a href="#column-profiles-userid">profiles.userid</a>) and an
email address (<a
href="#column-profiles-login_name">profiles.login_name</a>).</p>

<h3><a id="notes-authentication" name="notes-authentication">Authentication</a></h3>

<p>Each user has a password, used to authenticate that user to
Bugzilla.  The password is stored in <a
href="#column-profiles-cryptpassword">profiles.cryptpassword</a> in
encrypted form.  Before 2.14, it was also stored in <a
href="#column-profiles-password">profiles.password</a> as
plaintext.</p>

<p>On a successful login, Bugzilla generates a pair of cookies for the
user's browser.  On subsequent accesses, a user gets access if these
cookie checks pass (CGI.pl):</p>

<ol>

  <li>they have both Bugzilla_login and Bugzilla_logincookie cookies;</li>

  <li>their Bugzilla_login is the <a
  href="#column-profiles-login_name">login_name</a> of a row in the <a
  href="#table-profiles">profiles</a> table;</li>

  <li>their Bugzilla_logincookie matches a row in the <a
href="#table-logincookies">logincookies</a> table</li>

  <li>the userids of these two rows match;</li>

  <li>(before 2.16) the cryptpasswords of these two rows match;</li>

  <li>(before 2.14.2) the hostname of the logincookies row matches the
  CGI REMOTE_HOST;</li>

  <li>(from 2.14.2) the IP address of the logincookies row matches the
  CGI REMOTE_ADDR.</li>

</ol>

<p>and also their account is not disabled (i.e. <a
href="#column-profiles-disabledtext">profiles.disabledtext</a> is
empty).</p>

<p>If the cookie checks fail, the user has to login (with their
password), in which case a new row is added to the <a
href="#table-logincookies">logincookies</a> table and the user gets a
new pair of cookies.</p>

<p>Rows in the <a href="#table-logincookies">logincookies</a> table
are deleted after 30 days (at user login time; CGI.pl).</p>

<h3><a id="notes-voting" name="notes-voting">Voting</a></h3>

<p>Users may vote for bugs which they think are important.  The
maximum number of votes per user is product-dependent.  A user can
vote for a bug more than once.  Whether or not project managers pay
any attention to votes is up to them, apart from the "confirmation by
acclamation" process, which is as follows:</p>

<p>New bugs have the status UNCONFIRMED.  To enter the main workflow,
they need the status NEW.  To get the status NEW, they need a
particular number of votes which is product-dependent.</p>

<p>Votes are recorded in the <a href="#table-votes">votes table.</a></p>

<h3><a id="notes-milestones" name="notes-milestones">Milestones</a></h3>

<p>Products may have "milestones" defined.  The intention is that a
milestone should be a point in a project at which a set of bugs has
been resolved.  An example might be a product release or a QA target.
Milestones may be turned on and off with the parameter
"usetargetmilestone".</p>

<p>If milestones are on, each bug has a "target milestone" (by which
it should be fixed).  A product may have a URL associated with it
which locates a document describing the milestones for that product.
This document itself is entirely outside Bugzilla.  A product may also
have a default target milestone, which is given to new bugs.</p>

<p>Milestones for a product have a "sort key", which allows them to be
kept in an order.  As far as I can tell, this order is only used for
ordering user interface items (e.g. menu entries).</p>

<p>Milestones are kept in the <a href="#table-milestones">milestones</a> table.</p>

<h3><a id="notes-versions" name="notes-versions">Versions</a></h3>

<p>Products may have versions.  This allows more accurate bug
reporting: "we saw it in 1.3.7b3".  Versions are totally independent
of milestones.</p>

<h3><a id="notes-parameters" name="notes-parameters">Parameters</a></h3>

<p>The operation of Bugzilla is controlled by parameters.  These are
defined in defparams.pl, and set in editparams.cgi.  The current
values are stored in data/params.  They are <b>not</b> stored in the
database.</p>

<h3><a id="notes-groups" name="notes-groups">Groups</a></h3>

<p>Bugzilla has "groups" of users.  Membership of a group allows a
user to perform certain tasks.  Each group is represented by a row of
the <a href="#table-groups">groups</a> table.</p>

<p>There are a number of built-in groups, as follows:</p>

<table border="1" cellspacing="0" cellpadding="5">
  <tr align="left" valign="top">
  
    <th>Name</th>
  
    <th>Description</th>

  </tr>

  <tr align="left" valign="top">

    <td>tweakparams</td>

    <td>Can tweak operating parameters</td>

  </tr>

  <tr align="left" valign="top">

    <td>editusers</td>

    <td>Can edit or disable users</td>

  </tr>

  <tr align="left" valign="top">

    <td>creategroups</td>

    <td>Can create and destroy groups</td>

  </tr>

  <tr align="left" valign="top">

    <td>editcomponents</td>

    <td>Can create, destroy, and edit components</td>

  </tr>

  <tr align="left" valign="top">

    <td>editkeywords</td>

    <td>Can create, destroy, and edit keywords</td>

  </tr>

  <tr align="left" valign="top">

    <td>editbugs</td>

    <td>Can edit all aspects of any bug</td>

  </tr>

  <tr align="left" valign="top">

    <td>canconfirm</td>

    <td>Can confirm a bug</td>

  </tr>
</table>

<p>Group membership is conferred by the bit being set in <a
href="#column-profiles-groupset">profiles.groupset</a>.  Group
membership for new users is determined by matching <a
href="#column-groups-userregexp">groups.userregexp</a> against the new
user's email address.  The default configuration has universal regexps
for the "editbugs" and "canconfirm" groups.</p>

<p>New groups may be added (editgroups.cgi) and used to control access
to sets of bugs.  These "bug groups" have <a
href="#column-groups-isbuggroup">group.isbuggroup</a> set to 1.
Members of a bug group may see bugs in this group (using <a
href="#column-bugs-groupset">bugs.groupset</a>).</p>

<p>If the parameter "usebuggroups" is on, each product automatically
has a bug group associated with it.</p>

<p>If the parameter "usebuggroupsentry" is also on, the product's bug
group also determines the set of users able to enter bugs for the
product.</p>

<h3><a id="notes-attachments" name="notes-attachments">Attachments</a></h3>

<p>Users can upload attachments to bugs.  An attachments can be marked
as a patch.  Attachments are stored in the <a
href="#table-attachments">attachments</a> table.  From 2.16,
attachments can be marked as "obsolete".</p>

<p>From 2.16, each attachment may have one of a number of "status"
keywords associated with it.  The status keywords are user-defined on
a per-product basis.  The set of status keywords is defined in the <a
href="#table-attachstatusdefs">attachstatusdefs</a> table.  Whether a
given attachment has a given status keyword is defined by the <a
href="#table-attachstatuses">attachstatuses</a> table.</p>

<h3><a id="notes-keywords" name="notes-keywords">Keywords</a></h3>

<p>Bugzilla users can define a number of keywords, and then give each
bug a set of keywords.  This is mainly for use in finding related
bugs.  The keywords are stored in the <a
href="#table-keyworddefs">keyworddefs</a> table, and the one-to-many
mapping from bugs to keywords is stored in the <a
href="#table-keywords">keywords</a> table, and also in <a
href="#column-bugs-keywords">bugs.keywords</a>.</p>

<h3><a id="notes-dependencies" name="notes-dependencies">Dependencies</a></h3>

<p>Bugs may depend on other bugs being fixed.  That is, it may be
impossible to fix one bug until another one is fixed.  Bugzilla
records and displays such information and uses it to notify users when
a bug changes (all contacts for all dependent bugs are
notified when a bug changes).</p>

<p>Dependencies are recorded in the <a
href="#table-dependencies">dependencies</a>table.</p>

<h3><a id="notes-activity" name="notes-activity">Activity</a></h3>

<p>Bugzilla keeps a record of changes made to bugs.  This record is in
the <a href="#table-bugs_activity">bugs_activity</a> table.  Each row
in this table records a change to a field in the <a
href="#table-bugs">bugs</a> table.  The fields are referred to by a
number which is looked up in the <a
href="#table-fielddefs">fielddefs</a> table.  This table records the
name of the field and also a longer description used to display
activity tables.</p>

<h3><a id="notes-severity" name="notes-severity">Severity</a></h3>

<p>Each bug has a "severity" field, <a
href="#column-bugs-bug_severity">bugs.bug_severity</a>, indicating the
severity of the impact of the bug.  There is no code in Bugzilla which
distinguishes the values of this field, although it may naturally be
used in queries.  The intended meanings of the values of this field
are as follows:</p>

<table border="1" cellspacing="0" cellpadding="5">
  <tr align="left" valign="top">
  
    <th>Value</th>
  
    <th>Intended meaning</th>

  </tr>

  <tr align="left" valign="top">

    <td>Blocker</td>

    <td>Blocks development and/or testing work</td>

  </tr>

  <tr align="left" valign="top">

    <td>Critical</td>

    <td>Crashes, loss of data, severe memory leak</td>

  </tr>

  <tr align="left" valign="top">

    <td>Major</td>

    <td>Major loss of function</td>

  </tr>

  <tr align="left" valign="top">

    <td>Minor</td>

    <td>Minor loss of function, or other problem where easy workaround is present</td>

  </tr>

  <tr align="left" valign="top">

    <td>Trivial</td>

    <td>Cosmetic problem</td>

  </tr>

  <tr align="left" valign="top">

    <td>Enhancement</td>

    <td>Request for enhancement</td>

  </tr>
</table>

<h3><a id="notes-email" name="notes-email">Email notification</a></h3>

<p>When a bug changes, email notification is sent out to a number of
users:</p>

<ul>
  <li>The bug's owner (<a href="#column-bugs-assigned_to">bugs.assigned_to</a>)</li>
  <li>The bug's reporter (<a href="#column-bugs-reporter">bugs.reporter</a>)</li>
  <li>The bug's QA contact, if the "useqacontact" parameter is set (<a href="#column-bugs-qa_contact">bugs.qa_contact</a>)</li>
  <li>All the users who have explicitly asked to be notified when the bug changes (these users are stored in the <a href="#table-cc">cc</a> table).</li>
  <li>All the users who have voted for this bug (recorded in the <a href="#table-votes">votes</a> table).</li>
</ul>

<p>This is handled by the "processmail" script.</p>

<h3><a id="notes-descriptions" name="notes-descriptions">Long descriptions</a></h3>

<p>Each bug has a number of comments associated with it.  These are
stored in the <a href="#table-longdescs">longdescs</a> table and
displayed as the "Description" on the bug form, ordered by date and
annotated with the user and date.  Users can add new comments with the
"Additional comment" field on the bug form.</p>

<h3><a id="notes-namedqueries" name="notes-namedqueries">Named queries</a></h3>

<p>Users can name queries.  Named queries appear in a pop-up on the
query page.  A query named "(Default query)" is a user's default
query.  Named queries are stored in the <a
href="#table-namedqueries">namedqueries</a> table.</p>

<h3><a id="notes-watchers" name="notes-watchers">Watchers</a></h3>

<p>Bugzilla lets users "watch" each other; receiving each other's
Bugzilla email.  For instance, if Sam goes on holiday, Phil can
"watch" her, receiving all her Bugzilla email.  This is set up by the
user preferences (userprefs.cgi), recorded in the <a
href="#table-watch">watch</a> table and handled by the <a
href="#notes-email">processmail</a> script.</p>

<h3><a id="notes-shadow" name="notes-shadow">Shadow database</a></h3>

<p><b>Note: </b> Shadow databases were removed in Bugzilla 2.17.3.
Administrators can use MySQL's replication facilities to provide a
similar benefit.</p>

<p>Until Bugzilla 2.17.3, Bugzilla could maintain a shadow, read-only
copy of everything in another database (with the parameter
"shadowdb").  If the parameter "queryagainstshadowdb" is on, queries
were run against the shadow.  The <a
href="#table-shadowlog">shadowlog</a> table kept a record of SQL
activity since the last reflection.</p>

<h3><a id="notes-tables" name="notes-tables">List of tables</a></h3>

<table border="1" cellspacing="0" cellpadding="5">
  <tr valign="top" align="left">
  
    <th>Name</th>
  
    <th>Description</th>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-attachments">attachments</a></td>

    <td>Bug <a href="#notes-attachments">attachments</a>.</td>

  </tr>

  <tr bgcolor="#ffcccc" valign="top" align="left">

    <td><a href="#table-attachstatusdefs">attachstatusdefs</a></td>

    <td>Attachment status definitions <b>Added in 2.16, removed
    in 2.17.1.</b></td>

  </tr>

  <tr bgcolor="#ffcccc" valign="top" align="left">

    <td><a href="#table-attachstatuses">attachstatuses</a></td>

    <td>Attachment statuses  <b>Added in 2.16, removed
    in 2.17.1.</b>.</td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-bug_group_map">bug_group_map</a></td>

    <td><b>Added in 2.17.1</b>.</td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-bugs">bugs</a></td>

    <td>The bugs themselves.</td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-bugs_activity">bugs_activity</a></td>

    <td><a href="#notes-activity">Activity</a> on the bugs table.</td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-cc">cc</a></td>

    <td>Users who have asked to receive <a href="#notes-email">email</a> when a bug changes.</td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-components">components</a></td>

    <td>Components of <a href="#notes-products">products</a></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-dependencies">dependencies</a></td>

    <td>Which bugs <a href="#notes-dependencies">depend</a> on other bugs</td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-duplicates">duplicates</a></td>

    <td>Which bugs are duplicates of which other bugs.
    <b>Added in 2.12.</b></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-fielddefs">fielddefs</a></td>

    <td>Descriptions of fields in the <a href="#table-bugs">bugs</a> table.  Used for reporting <a href="#notes-activity">activity</a>.</td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-flagexclusions">flagexclusions</a></td>

    <td><b>Added in 2.17.1.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-flaginclusions">flaginclusions</a></td>

    <td><b>Added in 2.17.1.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-flags">flags</a></td>

    <td><b>Added in 2.17.1.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-flagtypes">flagtypes</a></td>

    <td><b>Added in 2.17.1.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-group_control_map">group_control_map</a></td>

    <td><b>Added in 2.17.3.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-group_group_map">group_group_map</a></td>

    <td><b>Added in 2.17.1.</b></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-groups">groups</a></td>

    <td>User <a href="#notes-groups">groups</a></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-keyworddefs">keyworddefs</a></td>

    <td>Descriptions of <a href="#notes-keywords">keywords</a>.</td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-keywords">keywords</a></td>

    <td>Which bugs have which <a href="#notes-keywords">keywords</a>.</td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-logincookies">logincookies</a></td>

    <td>Record of cookies used for <a href="#notes-authentication">authentication</a></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-longdescs">longdescs</a></td>

    <td><a href="#notes-descriptions">Long bug descriptions</a></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-milestones">milestones</a></td>

    <td><a href="#notes-milestones">Milestones</a></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-namedqueries">namedqueries</a></td>

    <td><a href="#notes-namedqueries">Named queries</a></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-products">products</a></td>

    <td><a href="#notes-products">Products</a></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-profiles">profiles</a></td>

    <td>The table of Bugzilla <a href="#notes-users">users</a>.</td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-profiles_activity">profiles_activity</a></td>

    <td>A record of activity in the profiles table.</td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-quips">quips</a></td>

    <td><b>Added in 2.17.1.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-series">series</a></td>

    <td><b>Added in 2.17.5.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-series_categories">series_categories</a></td>

    <td><b>Added in 2.17.5.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-series_data">series_data</a></td>

    <td><b>Added in 2.17.5.</b></td>

  </tr>

  <tr bgcolor="#ffcccc" valign="top" align="left">

    <td><a href="#table-shadowlog">shadowlog</a></td>

   <td>A log of SQL activity; used for updating shadow
    databases. <b>Removed in 2.17.3.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-tokens">tokens</a></td>

    <td>Email tokens. <b>Added in 2.14.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-user_group_map">user_group_map</a></td>

    <td><b>Added in 2.17.1.</b></td>

  </tr>

  <tr bgcolor="#ccffcc" valign="top" align="left">

    <td><a href="#table-user_series_map">user_series_map</a></td>

    <td><b>Added in 2.17.5.</b></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-versions">versions</a></td>

    <td>Product versions.</td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-votes">votes</a></td>

    <td><a href="#notes-voting">votes</a></td>

  </tr>

  <tr valign="top" align="left">

    <td><a href="#table-watch">watch</a></td>

    <td><a href="#notes-watchers">watchers</a></td>

  </tr>
</table>

<h2><a id="section-3" name="section-3">3. The schema</a></h2>

<p><b>Note: this schema description is now automatically generated.</b></p>

"""

# This afterword is included in the generated schema doc after the
# schema itself.

def afterword(credit_text):
    return """

<h2><a id="section-4" name="section-4">4. Example queries</a></h2>

<p>To select a bug given the bug number:</p>

<blockquote><code>
select * from bugs where bug_id = %d
</code></blockquote>

<p>To select the long descriptions of a bug, together with the name
and email address of the commenters:</p>

<blockquote><code>
select profiles.login_name, profiles.realname,
       longdescs.bug_when, longdescs.thetext
  from longdescs, profiles
 where profiles.userid = longdescs.who
   and longdescs.bug_id = %d
 order by longdescs.bug_when
</code></blockquote>

<p>To get a complete list of user ids and email addresses:</p>

<blockquote><code>
select userid, login_name from profiles
</code></blockquote>

<p>To find out a user's groupset:</p>

<blockquote><code>
select groupset from profiles where userid = %d
</code></blockquote>

<p>To get a user's email address:</p>

<blockquote><code>
select login_name from profiles where userid = %d
</code></blockquote>

<p>To get the set of cc addresses for a given bug:</p>

<blockquote><code>
select login_name from cc, profiles
 where cc.bug_id = %d
   and profiles.userid = cc.who
</code></blockquote>

<h2><a id="section-A" name="section-A">A. References</a></h2>

<table>

  <tr valign="top">

    <td>[<a id="ref-RB-2000-08-30" name="ref-RB-2000-08-30" href="/project/p4dti/doc/2000-08-30/design-document-structure/">RB 2000-08-30</a>]</td>

    <td>
    "Design document structure" (e-mail message);
    <a href="mailto:rb@ravenbrook.com">Richard Brooksby</a>;
    <a href="http://www.ravenbrook.com/">Ravenbrook Limited</a>;
    2000-08-30.
  </td>

  </tr>

  <tr valign="top">

    <td>[<a id="ref-RB-2000-10-05" name="ref-RB-2000-10-05" href="/project/p4dti/procedure/design-document/">RB 2000-10-05</a>]</td>

    <td>
    "P4DTI Project Design Document Procedure";
    <a href="mailto:rb@ravenbrook.com">Richard Brooksby</a>;
    <a href="http://www.ravenbrook.com/">Ravenbrook Limited</a>;
    2000-10-05.
  </td>

  </tr>

</table>


<h2><a id="section-B" name="section-B">B. Document History</a></h2>

<table>

  <tr valign="top">

    <td>2000-11-14</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Created.</td>

  </tr>

  <tr valign="top">

    <td> 2001-03-02 </td>

    <td> <a href="mailto:rb@ravenbrook.com">RB</a> </td>

    <td> Transferred copyright to Perforce under their license. </td>

  </tr>

  <tr valign="top">

    <td> 2001-04-06 </td>

    <td> <a href="mailto:nb@ravenbrook.com">NB</a> </td>

    <td> Added sample queries. </td>

  </tr>

  <tr valign="top">

    <td>2001-09-12</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Updated to reflect schema updates in Bugzilla 2.12 and 2.14</td>

  </tr>

  <tr valign="top">

    <td>2002-01-31</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Added notes on Bugzilla 2.14.1.</td>

  </tr>

  <tr valign="top">

    <td>2002-05-31</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Updated for Bugzilla 2.16 (based on 2.16rc1).</td>

  </tr>

  <tr valign="top">

    <td>2002-09-26</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Updated for Bugzilla 2.16/2.14.2/2.14.3.</td>

  </tr>

  <tr valign="top">

    <td>2002-10-04</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Added notes on Bugzilla 2.14.4 and 2.16.1, and on identical schemas.</td>

  </tr>

  <tr valign="top">

    <td>2003-05-14</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Added extensive notes on schema changes, in section 2.</td>

  </tr>

  <tr valign="top">

    <td>2003-06-06</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Added table of Bugzilla releases showing release date and support status.</td>

  </tr>

  <tr valign="top">

    <td>2003-06-06</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Added notes on schema changes in 2.17.x.</td>

  </tr>

  <tr valign="top">

    <td>2003-06-13</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Added first cut at description of new Bugzilla tables.</td>

  </tr>

  <tr valign="top">

    <td>2003-06-27</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Added more on recent schema changes.  Colour-coded all schema
  changes.</td>

  </tr>

  <tr valign="top">

    <td>2003-07-09</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Completely changed the way this document is produced.  The
    schema tables themselves are now created and coloured
    automatically by querying MySQL.</td>

  </tr>

  <tr valign="top">

    <td>2003-11-04</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Add Bugzilla 2.16.4 and 2.17.5.</td>

  </tr>

  <tr valign="top">

    <td>2003-11-10</td>

    <td><a href="mailto:nb@ravenbrook.com">NB</a></td>

    <td>Add Bugzilla 2.17.6.</td>

  </tr>

</table>

<hr />

<p> <small>This document is copyright &copy; 2001-2003 Perforce Software, Inc.  All rights reserved.</small> </p>

<p> <small>Redistribution and use of this document in any form, with or without modification, is permitted provided that redistributions of this document retain the above copyright notice, this condition and the following disclaimer.</small> </p>

<p> <small> <strong> This document is provided by the copyright holders and contributors "as is" and any express or implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed. In no event shall the copyright holders and contributors be liable for any direct, indirect, incidental, special, exemplary, or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this document, even if advised of the possibility of such damage. </strong> </small> </p>

<div align="center">

<p><code>""" + credit_text + """</code></p>

<p>
<a href="/">Ravenbrook</a> /
<a href="/project/">Projects</a> /
<a href="/project/p4dti/">Perforce Defect Tracking Integration</a> /
<a href="/project/p4dti/version/2.1/">Version 2.1 Product Sources</a> /
<a href="/project/p4dti/version/2.1/design/">Design</a>
</p>

</div>

</body>

</html>
"""

remarks_id = '$Id: //info.ravenbrook.com/project/p4dti/version/2.1/design/bugzilla-schema/schema_remarks.py#2 $'

# A. REFERENCES
#
#
# B. DOCUMENT HISTORY
#
# 2003-07-08 NB Created.
# 2003-07-09 NB See the history section of the "afterword" for
#               subsequent history items.
# 
#
# C. COPYRIGHT AND LICENSE
#
# This file is copyright (c) 2003 Perforce Software, Inc.  All rights
# reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
# 1.  Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#
# 2.  Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in
#     the documentation and/or other materials provided with the
#     distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# HOLDERS AND CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
# OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
# TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
# USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
# DAMAGE.
#
#
# $Id: //info.ravenbrook.com/project/p4dti/version/2.1/design/bugzilla-schema/schema_remarks.py#2 $
